package com.xebialabs.deployit.security.sql

import java.util
import com.xebialabs.deployit.core.sql.spring.Setter
import com.xebialabs.deployit.core.sql.{ColumnName, OrderBy, Queries, SchemaInfo, SelectBuilder, SqlFunction, TableName, SqlCondition => cond}
import com.xebialabs.deployit.engine.api.dto.{Ordering, Paging}
import com.xebialabs.deployit.security.model.{Tokens, XldUserToken}
import com.xebialabs.deployit.security.repository.XldUserTokenRespository
import com.xebialabs.deployit.security.sql.XldUserTokenSchema._
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.context.annotation.{Scope, ScopedProxyMode}
import org.springframework.jdbc.core.{JdbcTemplate, ResultSetExtractor}
import org.springframework.jdbc.support.GeneratedKeyHolder
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional

import java.sql.Connection
import java.util.Date
import scala.jdk.CollectionConverters._


@Component
@Scope(proxyMode = ScopedProxyMode.TARGET_CLASS)
@Transactional("mainTransactionManager")
class SqlXldUserTokenRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate)
                                 (@Autowired @Qualifier("mainSchema") override implicit val schemaInfo: SchemaInfo)
  extends XldUserTokenRespository with SqlXldUserTokenRepositoryQueries with Logging {

  override def listUserTokenByUserName(username: String, paging: Paging, order: Ordering, tokenNote: String): util.List[XldUserToken] = {
    logger.info("Listing user tokens for user: " + username)

    val selectBuilder = new SelectBuilder(tableName)
      .select(USERNAME)
      .select(CREATED_DATE)
      .select(LAST_USED_DATE)
      .select(EXPIRY_DATE)
      .select(TOKEN_NOTE)
      .select(TOKEN_HASH)
      .select(CI_ID)
      .where(cond.equals(USERNAME, username.toLowerCase()))

    withTextContainFilter(tokenNote, selectBuilder)

    if (paging != null) {
      selectBuilder.showPage(paging.page, paging.resultsPerPage)
    }
    if (order != null) {
      val columnToSort = fieldMapping(order.field)
      selectBuilder.orderBy(
        if (order.isAscending) OrderBy.asc(columnToSort) else OrderBy.desc(columnToSort)
      )
    }
   jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), mapRowToUserToken)
  }

  private def withTextContainFilter(tokenNote: String, builder: SelectBuilder): Unit = {
    val collectionOfCond = Seq(
      Option(tokenNote).filter(_.nonEmpty).map(un => cond.like(SqlFunction.lower(TOKEN_NOTE), s"%${un.toLowerCase}%")),
    ).flatten

    if(collectionOfCond.nonEmpty)
      builder.where(cond.or(collectionOfCond))
  }

  private val fieldMapping: Map[String, ColumnName] = Map(
    "tokenNote" -> TOKEN_NOTE,
    "expiryDate" -> EXPIRY_DATE,
    "createdDate" -> CREATED_DATE,
    "lastUse" -> LAST_USED_DATE,
    "expiresIn" -> EXPIRY_DATE
  )

  private def mapRowToUserToken: ResultSetExtractor[util.List[XldUserToken]] = rs => {
    val userTokenMap = scala.collection.mutable.Map[String, List[Tokens]]()
    while (rs.next) {
      val username = rs.getString(USERNAME.name)
      val token = Tokens(
        rs.getTimestamp(CREATED_DATE.name),
        rs.getTimestamp(LAST_USED_DATE.name),
        rs.getTimestamp(EXPIRY_DATE.name),
        rs.getString(TOKEN_NOTE.name),
        rs.getString(TOKEN_HASH.name),
        rs.getInt(CI_ID.name)
      )
      userTokenMap.update(username, token :: userTokenMap.getOrElse(username, Nil))
    }
    userTokenMap.map {
      case (username, tokens) =>   XldUserToken(username, tokens.reverse)
    }.toList.asJava
  }

  override def createUserToken(username: String, createdDate: Date, expiryDate: Date, tokenNote: String, tokenHash: String): Int = {
    val keyHolder = new GeneratedKeyHolder()
    jdbcTemplate.update((con: Connection) => {
      val preparedStatement = con.prepareStatement(INSERT, Array(XldUserTokenSchema.CI_ID.name))
      preparedStatement.setString(1, username.toLowerCase())
      preparedStatement.setString(2, tokenNote)
      preparedStatement.setString(3, tokenHash)
      preparedStatement.setTimestamp(4, new java.sql.Timestamp(createdDate.getTime))
      preparedStatement.setTimestamp(5, if (expiryDate != null) new java.sql.Timestamp(expiryDate.getTime) else null)
      preparedStatement
    }, keyHolder)
    logger.info("Created token for the user " + username + " with token note: " + tokenNote)
    keyHolder.getKey.intValue()
  }

  override def deleteTokenByUsername(username: String): Unit = {
    logger.info("Deleting user token for user: " + username)
    jdbcTemplate.update(DELETE, username.toLowerCase())
  }

  override def deleteTokenByCiIdAndUsername(username: String, ciId: Integer): Unit = {
    val i = jdbcTemplate.update(DELETE_WITH_CI_ID_AND_USERNAME, ciId, username)
    logger.info("Deleted " + i + " user token for ciId: " + ciId + " and username: " + username)
  }
}


object XldUserTokenSchema {
  val tableName: TableName = TableName("XLD_USER_TOKENS")
  val USERNAME: ColumnName = ColumnName("USERNAME")
  val CI_ID: ColumnName = ColumnName("CI_ID")
  val TOKEN_NOTE: ColumnName = ColumnName("TOKEN_NOTE")
  val TOKEN_HASH: ColumnName = ColumnName("TOKEN_HASH")
  val CREATED_DATE: ColumnName = ColumnName("CREATED_DATE")
  val LAST_USED_DATE: ColumnName = ColumnName("LAST_USED_DATE")
  val EXPIRY_DATE: ColumnName = ColumnName("EXPIRY_DATE")

}

trait SqlXldUserTokenRepositoryQueries extends Queries {
  val INSERT = sqlb"insert into $tableName ($USERNAME, $TOKEN_NOTE, $TOKEN_HASH, $CREATED_DATE, $EXPIRY_DATE) values (?, ?, ?, ?, ?)"
  val DELETE = sqlb"delete from $tableName where $USERNAME = ?"
  val DELETE_WITH_CI_ID_AND_USERNAME = sqlb"delete from $tableName where $CI_ID = ? and $USERNAME = ?"
}

