package com.xebialabs.deployit.security.sql

import ai.digital.deploy.permissions.client.{ReferencedPermissionServiceClient, RolePrincipalsServiceClient}
import com.xebialabs.deployit.core.sql.{SchemaInfo, SelectBuilder, Selectable, SqlCondition => cond}
import com.xebialabs.deployit.security.Permissions.authenticationToPrincipals
import com.xebialabs.deployit.security._
import com.xebialabs.deployit.security.permission.{Permission, PlatformPermissions}
import org.springframework.security.core.Authentication

import java.util
import java.util.{Collections, UUID}
import scala.jdk.CollectionConverters._

trait SqlPermissionFilter {

  def referencedPermissionServiceClient: ReferencedPermissionServiceClient
  def rolePrincipalsServiceClient: RolePrincipalsServiceClient

  def checker: PermissionChecker

  def addReadPermission(builder: SelectBuilder, referenceColumn: Selectable)(implicit schemaInfo: SchemaInfo): Unit = {
    val allPrincipals = authenticationToPrincipals(Permissions.getAuthentication)
    val allRoles = getRoles
    if (!isAdmin(allPrincipals, allRoles.asJava, Permissions.getAuthentication)) {
      val roleIds = allRoles.map(r => UUID.fromString(r.getId)).toSet
      val permissionNames = Permission.getReadPermissions.asScala.map(_.getPermissionName).toList
      val principalRoleIds = rolePrincipalsServiceClient.read(
        allPrincipals.asScala.toList,
        null).map(_.role.id).toSet

      val references = referencedPermissionServiceClient.getReferencesForRoles(
        (roleIds ++ principalRoleIds).toList,
        permissionNames
      ).map(_.toString)
      if (references.nonEmpty)
        builder.where(cond.in(referenceColumn, references))
      else
        builder.where(cond.equals(referenceColumn, "global-non-match"))
    }
  }

  private[this] def getRoles: List[Role] = {
    val viewAsRoles: Option[List[Role]] =
      Option(SecurityServiceLocator.getViewAsData).flatMap { viewAsData =>
        // roles from viewAs
        Option(viewAsData.getRoles).map(_.asScala.toList).orElse(
          Option(viewAsData.getUser).map(rolesFor)
        )
      }
    val roles: List[Role] = viewAsRoles.getOrElse(
      // roles from logged in user.
      rolesFor(Permissions.getAuthentication)
    )
    roles
  }

  private[this] def rolesFor(authentication: Authentication): List[Role] = {
    rolePrincipalsServiceClient.read(
      Permissions.authenticationToPrincipals(authentication).asScala.toList,
      null
    ).groupBy(_.role).map {
      case (role, rolePrincipals) =>
        val principalNames = rolePrincipals.flatMap(_.principals).asJava
        new Role(role.id.toString, role.name, principalNames)
    }.toList
  }

  private[this] def isAdmin(allPrincipals: util.Collection[String], allRoles: util.List[Role], auth: Authentication) = {
    allPrincipals.contains(PermissionEnforcer.ROLE_ADMIN) || checker.checkPermission(Collections.singletonList(PlatformPermissions.ADMIN), "", allRoles, auth)
  }
}
