package com.xebialabs.deployit.security.client

import ai.digital.deploy.permissions.api.rest.dto.{RoleDto, RoleWithPermissionsDto}
import ai.digital.deploy.permissions.client.{GlobalPermissionsServiceClient, ReferencedPermissionServiceClient, PermissionServiceClient => PermissionServicePermissionServiceClient}
import ai.digital.deploy.permissions.exception.RoleNameNotFoundException
import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.core.sql.spring.transactional
import com.xebialabs.deployit.engine.api.dto
import com.xebialabs.deployit.engine.api.dto.Paging
import com.xebialabs.deployit.engine.spi.exception.DeployitException
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.core.{Directory, Securable}
import com.xebialabs.deployit.repository.internal.Root
import com.xebialabs.deployit.security._
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.deployit.security.sql._
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.security.core.Authentication
import org.springframework.stereotype.Component
import org.springframework.transaction.PlatformTransactionManager

import java.util.{Collections, UUID, HashSet => JHashSet}
import java.{lang, util}
import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}


trait PermissionService extends PermissionEditor with PermissionLister with PermissionChecker {
  def removeCiPermissions(CiPk: Number): Unit
}

@Component("deployPermissionServiceClient")
class PermissionServiceClient(
                               @Autowired val ciResolver: CiResolver,
                               @Autowired val roleServiceClient: RoleServiceClient,
                               @Autowired val permissionServiceClient: PermissionServicePermissionServiceClient,
                               @Autowired val referencedPermissionServiceClient: ReferencedPermissionServiceClient,
                               @Autowired val globalPermissionServiceClient: GlobalPermissionsServiceClient,
                               @Autowired @Qualifier("mainTransactionManager") val mainTransactionManager: PlatformTransactionManager
) extends PermissionService with Logging {
  private final val SECURABLE = Type.valueOf(classOf[Securable])

  def pathToId(path: String): String =
    if (path.startsWith("/")) path.substring(1) else path

  def parentId(id: String): Option[String] = if (id.indexOf('/') == -1) None else
    Some(id).map(i => i.substring(0, i.lastIndexOf('/')))

  override def removeCiPermissions(CiPk: Number): Unit =
    ciResolver.getDirectoryUuid(CiPk) match {
      case Some(uuid) => permissionServiceClient.removeForReference(UUID.fromString(uuid))
      case _ =>
    }

  private def checkApplicablePermissions(onConfigurationItem: String, permissionValues: util.Collection[Permission]): Unit = {
    val notApplicableTo = Permissions.isApplicableTo(permissionValues, onConfigurationItem)
    checkArgument(notApplicableTo.isEmpty, "The permissions %s are not applicable to [%s]", notApplicableTo, onConfigurationItem)
  }

  private def removePermissionsForReference(reference: UUID): Unit =
    permissionServiceClient.removeForReference(reference)

  override def editPermissions(onConfigurationItem: String, permissions: util.Map[Role, util.Set[Permission]]): Unit = {
    logger.info(s"Edit ci $onConfigurationItem to assign next permissions: $permissions")
    val convertedPermissions = permissions.asScala
    val permissionValues: util.Collection[Permission] = convertedPermissions.flatMap(_._2.asScala).asJavaCollection
    checkApplicablePermissions(onConfigurationItem, permissionValues)
    val permissionNamesByRoleName = convertedPermissions.map { case (role, ps) => role.getName -> ps.asScala.map(_.getPermissionName) }
    val resolvedCi = getCiResolverFromId(onConfigurationItem)
    val parentSecuredCiPk = resolvedCi match {
      case ci: ResolvedCi if permissionValues.isEmpty  =>
        ci.securedCiPk match {
          case Some(securedCi) if securedCi == ci.pk && permissions.isEmpty =>
            ci.parentPk.flatMap(ciResolver.getSecuredCiId)
          case _ => None
        }
      case _ => None
    }

    val parentSecuredDir = resolvedCi match {
      case ci: ResolvedCi if permissionValues.isEmpty =>
        ci.securedDirectoryRef match {
          case Some(securedDirRef) if securedDirRef == ci.directoryUuid.orNull && permissions.isEmpty =>
            ci.parentPk.flatMap(ciResolver.getSecuredDirectoryReference)
          case _ => None
        }
      case _ => None
    }

    transactional(mainTransactionManager) {
      val removeScenario = resolvedCi match {
        case ci: ResolvedCi =>
          if (permissionValues.isEmpty) {
            // if it is empty we have two options
            ci.securedDirectoryRef match {
              case Some(securedDirRef) if securedDirRef == ci.directoryUuid.orNull && permissions.isEmpty =>
                ciResolver.removeSecuredCi(ci, parentSecuredCiPk)
                ciResolver.removeSecuredDirectoryReference(ci, parentSecuredDir)
                true
              case _ =>
                ciResolver.setSecuredCiId(ci)
                ciResolver.setSecuredDirectoryReference(ci)
                false
            }
          } else {
            ciResolver.setSecuredCiId(ci)
            ciResolver.setSecuredDirectoryReference(ci)
            false
          }
        case _ => false
      }

      val recordsByRoleName = resolveReference(resolvedCi).map(ref =>
        referencedPermissionServiceClient.readByRolePattern(ref, null).map(dto => (dto.role.name, dto.permissions)).groupBy(_._1).map {
          case (name, records) => (name, records.flatMap(_._2))
        }
      ).getOrElse(globalPermissionServiceClient.readByRolePattern(null).map(dto => (dto.role.name, dto.permissions)).groupBy(_._1).map {
        case (name, records) => (name, records.flatMap(_._2))
      })

      val toCreateRoleNames: Set[String] = permissionNamesByRoleName.keySet.toSet -- recordsByRoleName.keySet
      val createdPermissions: mutable.Map[String, List[String]] = mutable.Map[String, List[String]]()
      toCreateRoleNames.foreach { roleName =>
        val listOfPermissionNames = permissionNamesByRoleName(roleName).toList
        createdPermissions += (roleName -> listOfPermissionNames)
      }

      val deletedPermissions: mutable.Map[String, List[String]] = mutable.Map()
      recordsByRoleName.map { case (roleName, rows) =>
        val originalPermissionNames: Set[String] = rows.toSet
        val updatedPermissionsNames: Set[String] = permissionNamesByRoleName.getOrElse(roleName, Set()).toSet

        val toCreate = (updatedPermissionsNames -- originalPermissionNames).toList
        val toDelete = (originalPermissionNames -- updatedPermissionsNames).toList

        if (toCreate.nonEmpty) {
          createdPermissions += (roleName -> toCreate)
        }
        if (toDelete.nonEmpty) {
          deletedPermissions += (roleName -> toDelete)
        }
      }

      val recordsByRole = recordsByRoleName.map {
        case (roleName, _) => roleName
      }.toSet

      resolvedCi match {
        case ci: ResolvedCi if !permissionValues.isEmpty =>
          updatePermissionService(
            toCreateRoleNames ++ recordsByRole,
            ci.directoryUuid.map(UUID.fromString)
              .orElse(ci.directoryRef.map(UUID.fromString).orElse(None)),
            createdPermissions,
            deletedPermissions
          )
        case ci: ResolvedCi if permissionValues.isEmpty && removeScenario =>
          updatePermissionService(
            toCreateRoleNames ++ recordsByRole,
            parentSecuredDir.map(UUID.fromString).orElse(None),
            createdPermissions,
            deletedPermissions
          )
          ci.securedDirectoryRef.map(UUID.fromString).map(removePermissionsForReference)
        case ci: ResolvedCi =>
          updatePermissionService(
            toCreateRoleNames ++ recordsByRole,
            ci.directoryUuid.map(UUID.fromString).orElse(None),
            createdPermissions,
            deletedPermissions
          )
        case _ =>
          updatePermissionService(
            toCreateRoleNames ++ recordsByRole,
            None,
            createdPermissions,
            deletedPermissions
          )
      }
    }

  }

  private def updatePermissionService(recordsByRoleName: Set[String],
                                      directoryRef: Option[UUID],
                                      createdPermissions: mutable.Map[String, List[String]],
                                      deletedPermissions: mutable.Map[String, List[String]]): Unit =
    recordsByRoleName.foreach {
      roleName =>
        permissionServiceClient.createOrUpdate(
          directoryRef,
          roleName,
          createdPermissions.getOrElse(roleName, List()),
          deletedPermissions.getOrElse(roleName, List())
        )
    }

  private def getCiResolverFromId(onConfigurationItem: String, checkParent: Boolean = false): CiResolvedType =
    onConfigurationItem match {
      case GLOBAL_SECURITY_ALIAS | "" | null => GlobalResolvedCi
      case _ =>
        if (checkParent) {
          Try(ciResolver.getResolvedCiFromId(onConfigurationItem)) match {
            case Success(resolvedCi: ResolvedCi) => resolvedCi
            case Failure(e: NotFoundException) =>
              debug(s"ConfigurationItem $onConfigurationItem not found, resolving a parent")
              ciResolver.getResolvedCiFromId(parentId(onConfigurationItem).getOrElse(throw e))
          }
        }
        else {
          val resolvedCi = ciResolver.getResolvedCiFromId(onConfigurationItem)
          checkArgument(
            resolvedCi.ciType.instanceOf(SECURABLE),
            "The type [%s] does not support security permissions, because it doesn't implement com.xebialabs.deployit.repository.core.Securable",
            resolvedCi.ciType)
          resolvedCi
        }
    }

  private def resolveReference(onConfigurationItem: String, includeInherited: Boolean): Option[UUID] =
    resolveReference(getCiResolverFromId(onConfigurationItem, includeInherited))

  private def resolveReference(ciResolvedType: CiResolvedType): Option[UUID] =
    ciResolvedType match {
      case resolvedCi: ResolvedCi =>
        Option(UUID.fromString(ciResolver.getSecuredDirectoryReference(resolvedCi.pk).getOrElse(
          throw MissingCiReferenceException(s"Couldn't find ConfigurationItem [${resolvedCi.path}]")
        )))
      case _ =>
        None
    }

  private def checkPermissionsWithResolvedReference(onConfigurationItem: String, checkPermissionsCall: Option[UUID] => Boolean): Boolean = {
    Try(resolveReference(onConfigurationItem, includeInherited = true)) match {
      case Success(referenceMaybe) =>
        checkPermissionsCall(referenceMaybe)
      case Failure(e: NotFoundException) =>
        warn(e.getMessage)
        false
      case Failure(exception) => throw exception
    }
  }

  override def checkPermission(permissions: util.List[Permission], onConfigurationItem: String, allRoles: util.List[Role]): Boolean = {
    val permissionNames = permissions.asScala.map(_.getPermissionName).toList
    val roleNames = allRoles.asScala.map(_.getName).toList
    checkPermissionsWithResolvedReference(
      onConfigurationItem,
      referenceMaybe => permissionServiceClient.checkPermission(referenceMaybe, permissionNames, roleNames)
    )
  }

  override def checkPermission(permissions: util.List[Permission], onConfigurationItem: String, allRoles: util.List[Role], auth: Authentication): Boolean = {
    val permissionNames = permissions.asScala.map(_.getPermissionName).toList
    val roleNames = allRoles.asScala.map(_.getName).toList
    val principals = Permissions.authenticationToPrincipals(auth).asScala.toList
    checkPermissionsWithResolvedReference(
      onConfigurationItem,
      referenceMaybe => permissionServiceClient.checkPermission(referenceMaybe, permissionNames, roleNames, principals)
    )
  }

  private def getReferencesToPathMap(references: List[String]): Map[String, List[String]] = {
    val ciTypes = List(Type.valueOf(classOf[Root]), Type.valueOf(classOf[Directory]))
    ciResolver.getResolvedCiFromReferences(references, Option(ciTypes), securedDirOnly = true)
      .map(
        ci =>
          ci.securedDirectoryRef.getOrElse(
            throw MissingCiReferenceException(s"Cannot resolve CI reference for path: ${ci.path}")
          ) -> pathToId(ci.path)
      ).groupBy(_._1).map { case (k, v) => (k, v.map(_._2)) }
  }

  private def getReferenceToCiIdMap(onConfigurationItems: util.List[String]): Map[UUID, List[String]] =
    onConfigurationItems.asScala.map(id => {
      Try(resolveReference(id, includeInherited = true)) match {
        case Success(resolvedRef) => resolvedRef -> id
        case Failure(_: NotFoundException) => None -> id
        case Failure(exception) => throw exception
      }
    }).filter(_._1.isDefined)
      .map {
        case (resolvedRef, id) => resolvedRef.getOrElse(
          throw MissingCiReferenceException(s"Cannot resolve CI reference for path: $id")
        ) -> pathToId(id)
      }.groupBy(_._1).map { case (k, v) => (k, v.map(_._2).toList) }

  private def mapReferenceIdToCiId(referenceToIdMap: Map[UUID, List[String]], referenceId: String): List[String] =
    if (referenceId != GLOBAL_SECURITY_ALIAS) {
      referenceToIdMap.getOrElse(UUID.fromString(referenceId),
        throw MissingCiReferenceException(s"Cannot resolve CI ID reference mapping for: $referenceId")
      )
    }
    else {
      List(GLOBAL_SECURITY_ALIAS)
    }

  override def checkPermission(permissions: util.List[Permission], onConfigurationItems: util.List[String], allRoles: util.List[Role]): util.Map[String, lang.Boolean] = {
    val permissionNames = permissions.asScala.map(_.getPermissionName).toList
    val roleNames = allRoles.asScala.map(_.getName).toList
    val referenceToIdMap = getReferenceToCiIdMap(onConfigurationItems)
    val referenceOptions = referenceToIdMap.keys.toList
    permissionServiceClient.checkPermission(referenceOptions, permissionNames, roleNames)
      .map {
        case (referenceId, result) =>
        mapReferenceIdToCiId(referenceToIdMap, referenceId) -> lang.Boolean.valueOf(result)
      }
      .flatMap {
        case (paths, grant) => paths.map(_ -> grant)
      }.asJavaMutable()
  }

  override def checkPermission(permissions: util.List[Permission], onConfigurationItems: util.List[String], allRoles: util.List[Role], auth: Authentication): util.Map[String, lang.Boolean] = {
    val principals = Permissions.authenticationToPrincipals(auth).asScala.toList
    val permissionNames = permissions.asScala.map(_.getPermissionName).toList
    val roleNames = allRoles.asScala.map(_.getName).toList
    val referenceToIdMap = getReferenceToCiIdMap(onConfigurationItems)
    val referenceOptions = referenceToIdMap.keys.toList
    permissionServiceClient.checkPermission(referenceOptions, permissionNames, roleNames, principals)
      .map {
        case (referenceId, result) =>
          mapReferenceIdToCiId(referenceToIdMap, referenceId) -> lang.Boolean.valueOf(result)
      }
      .flatMap {
        case (paths, grant) => paths.map(_ -> grant)
      }.asJavaMutable()
  }

  private def mapRoleWithPermissions(
                                      roleWithPermissions: RoleWithPermissionsDto,
                                      referenceIdMap: Map[String, List[String]]
                                    ): Map[String, util.List[String]] = {
    roleWithPermissions.referencePermissions.groupBy(_.reference).map {
      case (reference, referencePermission) =>
        referenceIdMap.getOrElse(reference.toString,
          throw MissingCiReferenceException(s"Cannot resolve CI reference for path: $reference")
        ) -> referencePermission.flatMap(_.permissions).asJavaMutable()
    }.flatMap {
      case (paths, permissions) => paths.map(_ -> permissions)
    }
  }

  override def listPermissions(role: Role, paging: Paging): util.Map[String, util.List[String]] =
    Try {
      val roleWithPermissions = Option(paging).map(
        p => permissionServiceClient.getAllPermissionsForRole(
          role.getName,
          p.page,
          p.resultsPerPage
        )
      ).getOrElse(permissionServiceClient.getAllPermissionsForRole(role.getName))
      val references = roleWithPermissions.referencePermissions.map(_.reference).map(_.toString)
      val referenceIdMap = getReferencesToPathMap(references)
      val referencedPermissions = mapRoleWithPermissions(roleWithPermissions, referenceIdMap)
      (Map(GLOBAL_SECURITY_ALIAS -> roleWithPermissions.globalPermissions.asJavaMutable()) ++ referencedPermissions).asJavaMutable()
    } match {
      case Success(result) => result
      case Failure(_: RoleNameNotFoundException) =>
        warn(s"Could not find role [${role.getName}] - returning empty permissions")
        Collections.emptyMap()
    }

  override def listPermissions(roles: util.List[Role], paging: Paging): util.Map[String, util.List[String]] = {
    val rolesWithPermissions = Option(paging).map(
      p => permissionServiceClient.getAllPermissionsForRoles(
        roles.asScala.map(_.getName).toList,
        p.page,
        p.resultsPerPage
      ).data
    ).getOrElse(permissionServiceClient.getAllPermissionsForRoles(roles.asScala.map(_.getName).toList))
    val references = rolesWithPermissions.flatMap(_.referencePermissions.map(_.reference).map(_.toString))
    val referenceIdMap = getReferencesToPathMap(references)

    val referencedPermissions = rolesWithPermissions.flatMap(
      roleWithPermissions => mapRoleWithPermissions(roleWithPermissions, referenceIdMap)
    ).groupBy(_._1).map { case (key, value) => (key, value.map(_._2).flatMap(_.asScala).distinct.asJavaMutable())}
    val globalPermissions = rolesWithPermissions.flatMap(_.globalPermissions).distinct

    (Map(GLOBAL_SECURITY_ALIAS -> globalPermissions.asJavaMutable()) ++ referencedPermissions).asJavaMutable()
  }

  override def listGlobalPermissions(roles: util.List[Role], paging: Paging): util.Map[String, util.List[String]] = {
    val allPermissions =
      globalPermissionServiceClient.getGlobalPermissionsForRoles(roles.asScala.map(_.getName).toList)
        .flatMap(roleWithPermissions => {
          roleWithPermissions.permissions
        }).toSet
    if (allPermissions.nonEmpty) {
      Map(GLOBAL_SECURITY_ALIAS -> allPermissions.toList.asJavaMutable()).asJavaMutable()
    }
    else {
      Collections.emptyMap()
    }
  }

  private def convertToDeployRolePermissionMap(roleWithPermissions: List[(RoleDto, List[String])]): List[(Role, util.Set[Permission])] = {
    roleWithPermissions.map { case (role, permissions) =>
      val deployPermissions = permissions.flatMap(permissionName => Option(Permission.find(permissionName))).toSet.asJavaMutable()
      new Role(role.id.toString, role.name) -> deployPermissions
    }
  }

  override def readPermissions(onConfigurationItem: String,
                               rolePattern: String,
                               paging: Paging,
                               order: dto.Ordering,
                               includeInherited: lang.Boolean): util.Map[Role, util.Set[Permission]] = {
    Try(resolveReference(onConfigurationItem, includeInherited)) match {
      case Success(referenceMaybe) =>
        Option(paging).map(p =>
          referenceMaybe.map(referenceId => {
            convertToDeployRolePermissionMap(
              referencedPermissionServiceClient.read(
                referenceId,
                rolePattern,
                p.page,
                p.resultsPerPage,
                getSortOrder(order),
                "name"
              ).data
                .map(roleWithPermissions => (roleWithPermissions.role, roleWithPermissions.permissions))
            )
          }
          ).getOrElse(
            convertToDeployRolePermissionMap(
              globalPermissionServiceClient.read(
                rolePattern,
                p.page,
                p.resultsPerPage,
                getSortOrder(order),
                "name"
              ).data
                .map(roleWithPermissions => (roleWithPermissions.role, roleWithPermissions.permissions))
            )
          )
        ).getOrElse(
          referenceMaybe.map(referenceId =>
            convertToDeployRolePermissionMap(
              referencedPermissionServiceClient.readByRolePattern(
                referenceId,
                rolePattern
              ).map(roleWithPermissions => (roleWithPermissions.role, roleWithPermissions.permissions))
            )
          )
          .getOrElse(
            convertToDeployRolePermissionMap(
              globalPermissionServiceClient.readByRolePattern(rolePattern)
                .map(roleWithPermissions => (roleWithPermissions.role, roleWithPermissions.permissions))
            )
          )
        ).toMap.asJavaMutable()
      case Failure(exception) => throw exception
    }

  }

  private def getPermissionsForCiAndRole(id: String, role: Role): mutable.Set[String] =
    readPermissions(id, role.getName, null, null).asScala
      .getOrElse(role, new JHashSet[Permission]).asScala.map(_.getPermissionName)

  private def hasPermission(id: String, role: Role, permissionName: String): Boolean =
    getPermissionsForCiAndRole(id, role).contains(permissionName)

  override def grant(role: Role, permission: Permission, id: String): Unit = {
    if (!hasPermission(id, role, permission.getPermissionName)) {
      checkApplicablePermissions(id, List(permission).asJava)
      val roleName = role.getName
      logger.info(s"Granting $permission on $id for ${roleName}")
      val resolvedCi = getCiResolverFromId(id, true)
      transactional(mainTransactionManager) {
        resolvedCi match {
          case ci: ResolvedCi =>
            ciResolver.setSecuredCiId(ci)
            ciResolver.setSecuredDirectoryReference(ci)
          case _ =>
        }
        val permissionList = List(permission.getPermissionName)
        val permissions = mutable.Map(role.getName -> permissionList)
        resolvedCi match {
          case ci: ResolvedCi =>
            updatePermissionService(
              Set(role.getName),
              ci.directoryUuid.map(UUID.fromString)
                .orElse(ci.directoryRef.map(UUID.fromString).orElse(None)),
              permissions,
              mutable.Map.empty
            )
          case _ =>
            updatePermissionService(
              Set(role.getName),
              None,
              permissions,
              mutable.Map.empty
            )
        }
      }
    }
  }


  override def revoke(role: Role, permission: Permission, id: String): Unit = {
    val currentPermissions = getPermissionsForCiAndRole(id, role)
    if (currentPermissions.contains(permission.getPermissionName)) {
      logger.info(s"Revoking $permission on $id for ${role.getName}")
      val resolvedCi = getCiResolverFromId(id, true)
      val parentSecuredCiPk = resolvedCi match {
        case ci: ResolvedCi =>
          ci.securedCiPk match {
            case Some(securedCi) if securedCi == ci.pk =>
              ci.parentPk.flatMap(ciResolver.getSecuredCiId)
            case _ => None
          }
        case _ => None
      }
      val parentSecuredDir = resolvedCi match {
        case ci: ResolvedCi =>
          ci.securedDirectoryRef match {
            case Some(securedDirRef) if securedDirRef == ci.directoryUuid.orNull =>
              ci.parentPk.flatMap(ciResolver.getSecuredDirectoryReference)
            case _ => None
          }
        case _ => None
      }

      transactional(mainTransactionManager) {
        resolvedCi match {
          case ci: ResolvedCi if currentPermissions.size == 1 =>
            ciResolver.removeSecuredCi(ci, parentSecuredCiPk)
            ciResolver.removeSecuredDirectoryReference(ci, parentSecuredDir)
          case _ =>
        }
        val permissionList = List(permission.getPermissionName)
        val permissions = mutable.Map(role.getName -> permissionList)
        resolvedCi match {
          case ci: ResolvedCi =>
            updatePermissionService(
              Set(role.getName),
              ci.directoryUuid.map(UUID.fromString)
                .orElse(ci.directoryRef.map(UUID.fromString).orElse(None)),
              mutable.Map.empty,
              permissions
            )
          case _ =>
            updatePermissionService(
              Set(role.getName),
              None,
              mutable.Map.empty,
              permissions
            )
        }
      }
    }
  }

  override def updatePermissions(onConfigurationItem: String, addedPermissions: util.Map[Role, util.List[Permission]], removedPermissions: util.Map[Role, util.List[Permission]]): Unit = {
    logger.info(s"Edit ci $onConfigurationItem to assign next permissions: $addedPermissions and remove next permissions: $removedPermissions")
    val convertedAddedPermissions = addedPermissions.asScala
    val convertedRemovedPermissions = removedPermissions.asScala
    val addedPermissionValues: util.Collection[Permission] = convertedAddedPermissions.flatMap(_._2.asScala).asJavaCollection

    checkApplicablePermissions(onConfigurationItem, addedPermissionValues)

    val roleNames = addedPermissions.keySet().asScala.union(removedPermissions.keySet().asScala).map(_.getName).toSet
    val addedPermissionNamesByRoleName = convertedAddedPermissions.map { case (role, ps) => role.getName -> ps.asScala.map(_.getPermissionName).toList }
    val removedPermissionNamesByRoleName = convertedRemovedPermissions.map { case (role, ps) => role.getName -> ps.asScala.map(_.getPermissionName).toList }

    val resolvedCi = getCiResolverFromId(onConfigurationItem)
    val parentSecuredCiPk = resolvedCi match {
      case ci: ResolvedCi if addedPermissions.isEmpty && removedPermissions.isEmpty =>
        ci.securedCiPk match {
          case Some(securedCi) if securedCi == ci.pk && addedPermissions.isEmpty && removedPermissions.isEmpty =>
            ci.parentPk.flatMap(ciResolver.getSecuredCiId)
          case _ => None
        }
      case _ => None
    }

    val parentSecuredDir = resolvedCi match {
      case ci: ResolvedCi if addedPermissions.isEmpty && removedPermissions.isEmpty =>
        ci.securedDirectoryRef match {
          case Some(securedDirRef) if securedDirRef == ci.directoryUuid.orNull && addedPermissions.isEmpty && removedPermissions.isEmpty =>
            ci.parentPk.flatMap(ciResolver.getSecuredDirectoryReference)
          case _ => None
        }
      case _ => None
    }

    transactional(mainTransactionManager) {
      val removeScenario = resolvedCi match {
        case ci: ResolvedCi =>
          if (addedPermissions.isEmpty && removedPermissions.isEmpty) {
            // if it is empty we have two options
            ci.securedDirectoryRef match {
              case Some(securedDirRef) if securedDirRef == ci.directoryUuid.orNull && addedPermissions.isEmpty && removedPermissions.isEmpty =>
                ciResolver.removeSecuredCi(ci, parentSecuredCiPk)
                ciResolver.removeSecuredDirectoryReference(ci, parentSecuredDir)
                true
              case _ =>
                ciResolver.setSecuredCiId(ci)
                ciResolver.setSecuredDirectoryReference(ci)
                false
            }
          } else {
            ciResolver.setSecuredCiId(ci)
            ciResolver.setSecuredDirectoryReference(ci)
            false
          }
        case _ => false
      }

      resolvedCi match {
        case ci: ResolvedCi if !(addedPermissions.isEmpty && removedPermissions.isEmpty) =>
          updatePermissionService(
            roleNames,
            ci.directoryUuid.map(UUID.fromString)
              .orElse(ci.directoryRef.map(UUID.fromString).orElse(None)),
            addedPermissionNamesByRoleName,
            removedPermissionNamesByRoleName
          )
        case ci: ResolvedCi if addedPermissions.isEmpty && removedPermissions.isEmpty && removeScenario =>
          updatePermissionService(
            roleNames,
            parentSecuredDir.map(UUID.fromString).orElse(None),
            addedPermissionNamesByRoleName,
            removedPermissionNamesByRoleName
          )
          ci.securedDirectoryRef.map(UUID.fromString).map(removePermissionsForReference)
        case ci: ResolvedCi =>
          updatePermissionService(
            roleNames,
            ci.directoryUuid.map(UUID.fromString).orElse(None),
            addedPermissionNamesByRoleName,
            removedPermissionNamesByRoleName
          )
        case _ =>
          updatePermissionService(
            roleNames,
            None,
            addedPermissionNamesByRoleName,
            removedPermissionNamesByRoleName
          )
      }
    }
  }
}

final case class MissingCiReferenceException(msg: String) extends DeployitException(msg)
