package com.xebialabs.deployit.security.client

import ai.digital.deploy.permissions.api.rest.dto.{RoleDto, RoleWithPermissionsDto}
import ai.digital.deploy.permissions.client.{GlobalPermissionsServiceClient, ReferencedPermissionServiceClient, PermissionServiceClient => PermissionServicePermissionServiceClient}
import com.xebialabs.deployit.engine.api.dto
import com.xebialabs.deployit.engine.api.dto.Paging
import com.xebialabs.deployit.engine.spi.exception.DeployitException
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.deployit.security.sql._
import com.xebialabs.deployit.security.{GLOBAL_SECURITY_ALIAS, Permissions, Role}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.context.annotation.Primary
import org.springframework.security.core.Authentication
import org.springframework.stereotype.Component

import java.util.{Collections, UUID}
import java.{lang, util}
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

@Component
@Primary
@ConditionalOnProperty(name = Array("xl.permission-service.enabled"), havingValue = "true", matchIfMissing = true)
class PermissionServiceClient(
                               @Autowired @Qualifier("sqlPermissionService") permissionService: PermissionService,
                               @Autowired val ciResolver: CiResolver,
                               @Autowired val permissionServiceClient: PermissionServicePermissionServiceClient,
                               @Autowired val referencedPermissionServiceClient: ReferencedPermissionServiceClient,
                               @Autowired val globalPermissionServiceClient: GlobalPermissionsServiceClient
                             ) extends PermissionService with Logging {
  def pathToId(path: String): String =
    if (path.startsWith("/")) path.substring(1) else path

  def parentId(id: String): Option[String] = if (id.indexOf('/') == -1) None else
    Some(id).map(i => i.substring(0, i.lastIndexOf('/')))

  override def removeCiPermissions(CiPk: Number): Unit =
    permissionService.removeCiPermissions(CiPk)

  override def editPermissions(onConfigurationItem: String, permissions: util.Map[Role, util.Set[Permission]]): Unit =
    permissionService.editPermissions(onConfigurationItem, permissions)

  private def getCiResolverFromId(onConfigurationItem: String): CiResolvedType =
    onConfigurationItem match {
      case GLOBAL_SECURITY_ALIAS | "" | null => GlobalResolvedCi
      case _ =>
        Try(ciResolver.getResolvedCiFromId(onConfigurationItem)) match {
          case Success(resolvedCi: ResolvedCi) => resolvedCi
          case Failure(e: NotFoundException) =>
            debug(s"ConfigurationItem $onConfigurationItem not found, resolving a parent")
            ciResolver.getResolvedCiFromId(parentId(onConfigurationItem).getOrElse(throw e))
        }
    }

  private def resolveReference(onConfigurationItem: String) = {
    getCiResolverFromId(onConfigurationItem) match {
      case resolvedCi: ResolvedCi =>
        Option(UUID.fromString(ciResolver.getSecuredDirectoryReference(resolvedCi.pk).getOrElse(
          throw MissingCiReferenceException(s"Cannot resolve CI reference for path: $onConfigurationItem")
        )))
      case _ =>
        None
    }
  }

  private def checkPermissionsWithResolvedReference(onConfigurationItem: String, checkPermissionsCall: Option[UUID] => Boolean): Boolean = {
    Try(resolveReference(onConfigurationItem)) match {
      case Success(referenceMaybe) =>
        checkPermissionsCall(referenceMaybe)
      case Failure(e: NotFoundException) =>
        warn(e.getMessage)
        false
      case Failure(exception) => throw exception
    }
  }

  override def checkPermission(permissions: util.List[Permission], onConfigurationItem: String, allRoles: util.List[Role]): Boolean = {
    val permissionNames = permissions.asScala.map(_.getPermissionName).toList
    val roleNames = allRoles.asScala.map(_.getName).toList
    checkPermissionsWithResolvedReference(
      onConfigurationItem,
      referenceMaybe => permissionServiceClient.checkPermission(referenceMaybe, permissionNames, roleNames)
    )
  }

  override def checkPermission(permissions: util.List[Permission], onConfigurationItem: String, allRoles: util.List[Role], auth: Authentication): Boolean = {
    val permissionNames = permissions.asScala.map(_.getPermissionName).toList
    val roleNames = allRoles.asScala.map(_.getName).toList
    val principals = Permissions.authenticationToPrincipals(auth).asScala.toList
    checkPermissionsWithResolvedReference(
      onConfigurationItem,
      referenceMaybe => permissionServiceClient.checkPermission(referenceMaybe, permissionNames, roleNames, principals)
    )
  }

  private def getReferencesToPathMap(references: List[String]): Map[String, List[String]] =
    ciResolver.getResolvedCiFromReferences(references)
      .map(
        ci =>
          ci.securedDirectoryRef.getOrElse(
            throw MissingCiReferenceException(s"Cannot resolve CI reference for path: ${ci.path}")
          ) -> pathToId(ci.path)
      ).groupBy(_._1).map { case (k, v) => (k, v.map(_._2)) }

  private def getReferenceToCiIdMap(onConfigurationItems: util.List[String]): Map[UUID, List[String]] =
    onConfigurationItems.asScala.map(id => {
      Try(resolveReference(id)) match {
        case Success(resolvedRef) => resolvedRef -> id
        case Failure(_: NotFoundException) => None -> id
        case Failure(exception) => throw exception
      }
    }).filter(_._1.isDefined)
      .map {
        case (resolvedRef, id) => resolvedRef.getOrElse(
          throw MissingCiReferenceException(s"Cannot resolve CI reference for path: $id")
        ) -> pathToId(id)
      }.groupBy(_._1).map { case (k, v) => (k, v.map(_._2).toList) }

  private def mapReferenceIdToCiId(referenceToIdMap: Map[UUID, List[String]], referenceId: String): List[String] =
    if (referenceId != GLOBAL_SECURITY_ALIAS) {
      referenceToIdMap.getOrElse(UUID.fromString(referenceId),
        throw MissingCiReferenceException(s"Cannot resolve CI ID reference mapping for: $referenceId")
      )
    }
    else {
      List(GLOBAL_SECURITY_ALIAS)
    }

  override def checkPermission(permissions: util.List[Permission], onConfigurationItems: util.List[String], allRoles: util.List[Role]): util.Map[String, lang.Boolean] = {
    val permissionNames = permissions.asScala.map(_.getPermissionName).toList
    val roleNames = allRoles.asScala.map(_.getName).toList
    val referenceToIdMap = getReferenceToCiIdMap(onConfigurationItems)
    val referenceOptions = referenceToIdMap.keys.toList
    permissionServiceClient.checkPermission(referenceOptions, permissionNames, roleNames)
      .map {
        case (referenceId, result) =>
        mapReferenceIdToCiId(referenceToIdMap, referenceId) -> lang.Boolean.valueOf(result)
      }
      .flatMap {
        case (paths, grant) => paths.map(_ -> grant)
      }.asJavaMutable()
  }

  override def checkPermission(permissions: util.List[Permission], onConfigurationItems: util.List[String], allRoles: util.List[Role], auth: Authentication): util.Map[String, lang.Boolean] = {
    val principals = Permissions.authenticationToPrincipals(auth).asScala.toList
    val permissionNames = permissions.asScala.map(_.getPermissionName).toList
    val roleNames = allRoles.asScala.map(_.getName).toList
    val referenceToIdMap = getReferenceToCiIdMap(onConfigurationItems)
    val referenceOptions = referenceToIdMap.keys.toList
    permissionServiceClient.checkPermission(referenceOptions, permissionNames, roleNames, principals)
      .map {
        case (referenceId, result) =>
          mapReferenceIdToCiId(referenceToIdMap, referenceId) -> lang.Boolean.valueOf(result)
      }
      .flatMap {
        case (paths, grant) => paths.map(_ -> grant)
      }.asJavaMutable()
  }

  private def mapRoleWithPermissions(
                                      roleWithPermissions: RoleWithPermissionsDto,
                                      referenceIdMap: Map[String, List[String]]
                                    ): Map[String, util.List[String]] = {
    roleWithPermissions.referencePermissions.groupBy(_.reference).map {
      case (reference, referencePermission) =>
        referenceIdMap.getOrElse(reference.toString,
          throw MissingCiReferenceException(s"Cannot resolve CI reference for path: $reference")
        ) -> referencePermission.flatMap(_.permissions).asJavaMutable()
    }.flatMap {
      case (paths, permissions) => paths.map(_ -> permissions)
    }
  }

  override def listPermissions(role: Role, paging: Paging): util.Map[String, util.List[String]] = {
    val roleWithPermissions = Option(paging).map(
      p => permissionServiceClient.getAllPermissionsForRole(
        role.getName,
        p.page,
        p.resultsPerPage
      )
    ).getOrElse(permissionServiceClient.getAllPermissionsForRole(role.getName))
    val references = roleWithPermissions.referencePermissions.map(_.reference).map(_.toString)
    val referenceIdMap = getReferencesToPathMap(references)
    val referencedPermissions = mapRoleWithPermissions(roleWithPermissions, referenceIdMap)
    (Map(GLOBAL_SECURITY_ALIAS -> roleWithPermissions.globalPermissions.asJavaMutable()) ++ referencedPermissions).asJavaMutable()
  }

  override def listPermissions(roles: util.List[Role], paging: Paging): util.Map[String, util.List[String]] = {
    val rolesWithPermissions = Option(paging).map(
      p => permissionServiceClient.getAllPermissionsForRoles(
        roles.asScala.map(_.getName).toList,
        p.page,
        p.resultsPerPage
      ).data
    ).getOrElse(permissionServiceClient.getAllPermissionsForRoles(roles.asScala.map(_.getName).toList))
    val references = rolesWithPermissions.flatMap(_.referencePermissions.map(_.reference).map(_.toString))
    val referenceIdMap = getReferencesToPathMap(references)

    val referencedPermissions = rolesWithPermissions.flatMap(
      roleWithPermissions => mapRoleWithPermissions(roleWithPermissions, referenceIdMap)
    ).groupBy(_._1).map { case (key, value) => (key, value.map(_._2).flatMap(_.asScala).distinct.asJavaMutable())}
    val globalPermissions = rolesWithPermissions.flatMap(_.globalPermissions).distinct

    (Map(GLOBAL_SECURITY_ALIAS -> globalPermissions.asJavaMutable()) ++ referencedPermissions).asJavaMutable()
  }

  override def listGlobalPermissions(roles: util.List[Role], paging: Paging): util.Map[String, util.List[String]] =
    globalPermissionServiceClient.getGlobalPermissionsForRoles(roles.asScala.map(_.getName).toList)
      .map(roleWithPermissions => {
        roleWithPermissions.role.name -> roleWithPermissions.permissions.asJavaMutable()
      }).toMap.asJavaMutable()

  private def convertToDeployRolePermissionMap(roleWithPermissions: List[(RoleDto, List[String])]): List[(Role, util.Set[Permission])] = {
    roleWithPermissions.map { case (role, permissions) =>
      val deployPermissions = permissions.flatMap(permissionName => Option(Permission.find(permissionName))).toSet.asJavaMutable()
      new Role(role.id.toString, role.name) -> deployPermissions
    }
  }

  override def readPermissions(onConfigurationItem: String,
                               rolePattern: String,
                               paging: Paging,
                               order: dto.Ordering,
                               includeInherited: lang.Boolean): util.Map[Role, util.Set[Permission]] = {
    Try(resolveReference(onConfigurationItem)) match {
      case Success(referenceMaybe) =>
        Option(paging).map(p =>
          referenceMaybe.map(referenceId => {
            convertToDeployRolePermissionMap(
              referencedPermissionServiceClient.read(
                referenceId,
                rolePattern,
                p.page,
                p.resultsPerPage,
                getSortOrder(order),
                "name"
              ).data
                .map(roleWithPermissions => (roleWithPermissions.role, roleWithPermissions.permissions))
            )
          }
          ).getOrElse(
            convertToDeployRolePermissionMap(
              globalPermissionServiceClient.read(
                rolePattern,
                p.page,
                p.resultsPerPage,
                getSortOrder(order),
                "name"
              ).data
                .map(roleWithPermissions => (roleWithPermissions.role, roleWithPermissions.permissions))
            )
          )
        ).getOrElse(
          referenceMaybe.map(referenceId =>
            convertToDeployRolePermissionMap(
              referencedPermissionServiceClient.readByRolePattern(
                referenceId,
                rolePattern
              ).map(roleWithPermissions => (roleWithPermissions.role, roleWithPermissions.permissions))
            )
          )
            .getOrElse(
              convertToDeployRolePermissionMap(
                globalPermissionServiceClient.readByRolePattern(rolePattern)
                  .map(roleWithPermissions => (roleWithPermissions.role, roleWithPermissions.permissions))
              )
            )
        ).toMap.asJavaMutable()
      case Failure(_: NotFoundException) => Collections.emptyMap[Role, util.Set[Permission]]()
      case Failure(exception) => throw exception
    }

  }

  override def grant(role: Role, permission: Permission, id: String): Unit =
    permissionService.grant(role, permission, id)

  override def revoke(role: Role, permission: Permission, id: String): Unit =
    permissionService.revoke(role, permission, id)
}

final case class MissingCiReferenceException(msg: String) extends DeployitException(msg)
