package com.xebialabs.deployit.security.sql

import java.sql.ResultSet
import java.util
import com.xebialabs.deployit.core.sql.spring.Setter
import com.xebialabs.deployit.core.sql.{ColumnName, JoinBuilder, JoinType, Queries, SchemaInfo, SelectBuilder, TableName, SqlCondition => cond}
import com.xebialabs.deployit.security.model.{XldUserCredentials, XldUserProfile}
import com.xebialabs.deployit.security.repository.XldUserProfileRepository
import com.xebialabs.deployit.security.sql.SqlXldUserProfileRepository._
import com.xebialabs.deployit.security.sql.XldUserProfileSchema._
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.context.annotation.{Scope, ScopedProxyMode}
import org.springframework.jdbc.core.{JdbcTemplate, ResultSetExtractor, RowMapper}
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional

import scala.jdk.CollectionConverters._

@Component
@Scope(proxyMode = ScopedProxyMode.TARGET_CLASS)
@Transactional("mainTransactionManager")
class SqlXldUserProfileRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate)
                                 (@Autowired @Qualifier("mainSchema") override implicit val schemaInfo: SchemaInfo)
  extends XldUserProfileRepository with SqlXldUserProfileRepositoryQueries {

  override def createProfile(username: String, analyticsEnabled: Boolean): Unit = {
    jdbcTemplate.update(INSERT, username.toLowerCase(), analyticsEnabled)
  }

  override def updateProfile(username: String, analyticsEnabled: Boolean): Unit = {
    jdbcTemplate.update(UPDATE, analyticsEnabled, username.toLowerCase())
  }

  override def findOne(username: String, loadCredentials: Boolean = false): Option[XldUserProfile] = {
    val selectBuilder = new SelectBuilder(tableName)
      .where(cond.equals(USERNAME, username.toLowerCase))
      .as(userProfileTableAlias)
    if (!loadCredentials) {
      jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), mapRowToUserProfile)
    }.asScala.headOption else Option({
      val joinBuilder = new JoinBuilder(selectBuilder)
        .join(
          new SelectBuilder(XldUserCredentialsSchema.tableName).as(userCredentialsTableAlias),
          cond.equals(
            USERNAME.tableAlias(userProfileTableAlias),
            XldUserCredentialsSchema.PROFILE_USERNAME.tableAlias(userCredentialsTableAlias)
          ),
          JoinType.Left
        )
        .join(
          new SelectBuilder(XldUserDefaultCredentialsSchema.tableName).as(userDefaultCredentialsTableAlias),
          cond.equals(
            XldUserCredentialsSchema.ID.tableAlias(userCredentialsTableAlias),
            XldUserDefaultCredentialsSchema.USER_CREDENTIALS_ID.tableAlias(userDefaultCredentialsTableAlias)
          ),
          JoinType.Left
        )
      jdbcTemplate.query(joinBuilder.query, Setter(joinBuilder.parameters), mapRowToUserProfileWithCredentials)
    })
  }

  override def delete(username: String): Unit = {
    jdbcTemplate.update(DELETE, Setter(Seq(username.toLowerCase())))
  }

  private def mapRowToUserProfile: RowMapper[XldUserProfile] = (rs: ResultSet, _) =>
    XldUserProfile(rs.getString(USERNAME.name), rs.getBoolean(ANALYTICS_ENABLED.name))

  private def mapRowToUserProfileWithCredentials: ResultSetExtractor[XldUserProfile] = (rs: ResultSet) => {
    var row = 0
    val list = new util.ArrayList[(String, Boolean, XldUserCredentials)]()
    while (rs.next()) {
      list.add(rowMapper.mapRow(rs, row))
      row += 1
    }
    list.asScala.toList.groupBy(_._1).map {
      case (username, profileCredPairs) => XldUserProfile(username, profileCredPairs.head._2, profileCredPairs.map(_._3).toSet)
    }.headOption.orNull
  }

  private val rowMapper: RowMapper[(String, Boolean, XldUserCredentials)] = (rs: ResultSet, row: Int) => {
    (
      rs.getString(USERNAME.name),
      rs.getBoolean(ANALYTICS_ENABLED.name),
      userCredentialsRowMapper.mapRow(rs, row)
    )
  }

  private val userCredentialsRowMapper =
    SqlXldUserCredentialsRepository.createRowMapper()
}

object SqlXldUserProfileRepository {
  val userProfileTableAlias = "xup"
  val userCredentialsTableAlias = "xuc"
  val userDefaultCredentialsTableAlias = "xudc"
}

object XldUserProfileSchema {
  val tableName: TableName = TableName("XLD_USER_PROFILES")
  val USERNAME: ColumnName = ColumnName("USERNAME")
  val ANALYTICS_ENABLED: ColumnName = ColumnName("ANALYTICS_ENABLED")
}

trait SqlXldUserProfileRepositoryQueries extends Queries {
  val INSERT = sqlb"insert into $tableName ($USERNAME, $ANALYTICS_ENABLED) values (?, ?)"
  val UPDATE = sqlb"update $tableName set $ANALYTICS_ENABLED = ? where $USERNAME = ?"
  val DELETE = sqlb"delete from $tableName where ($USERNAME) = (?)"
}
