package com.xebialabs.deployit.repository.sql.base

import com.xebialabs.deployit.sql.base.schema.CIS
import com.xebialabs.deployit.core.sql.SqlFunction.?
import com.xebialabs.deployit.core.sql.{AscOrdering, ColumnName, JoinBuilder, Queries, SelectBuilder, SqlLiteral, TableName, SqlCondition => cond}

trait CiQueries extends Queries {
  lazy val INSERT: String = {
    import CIS._
    sqlb"insert into $tableName ($ci_type, $name, $token, $directory_uuid, $created_at, $created_by, $modified_at, $modified_by, $path, $scm_traceability_data_id, $reference_id) values (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"
  }

  lazy val UPDATE_PARENT: String = {
    import CIS._
    sqlb"update $tableName set $parent_id = ? where $ID = ?"
  }

  lazy val SELECT_CI_BY_ID: String = {
    import CIS._
    sqlb"select * from $tableName where $ID = ?"
  }

  lazy val SELECT_CI_PATH_BY_ID: String = {
    import CIS._
    sqlb"select $path from $tableName where $ID = ?"
  }

  lazy val SELECT_CI_BY_PATH: String = {
    import CIS._
    sqlb"select * from $tableName where $path = ?"
  }

  def buildSelectCisByPathsQuery(paths: Iterable[String]): String = {
    import CIS._
    new SelectBuilder(tableName).select(new SqlLiteral("*")).where(cond.in(path, paths)).query
  }

  lazy val SELECT_CI_BY_PARENT_ID: String = {
    import CIS._
    sqlb"select $ID, $ci_type, $path, $scm_traceability_data_id from $tableName where $parent_id = ?"
  }

  lazy val SELECT_ID_BY_PATH: String = {
    import CIS._
    sqlb"select $ID from $tableName where $path = ?"
  }

  def buildSelectIdsByPathsQuery(paths: Seq[String]): String = {
    import CIS._
    new SelectBuilder(tableName).select(ID).select(path).where(cond.in(path, paths)).query
  }

  def buildSelectIdsAndTokensByPathsQuery(paths: Seq[String]): String = {
    import CIS._
    new SelectBuilder(tableName).select(ID).select(path).select(token).where(cond.in(path, paths)).query
  }

  def buildSelectCisByRelationsTableQuery(joinTable: TableName,
                                          joinColumn: ColumnName,
                                          predicateColumn: ColumnName,
                                          orderByColumn: Option[ColumnName]): String = {
    val cisAlias = "cis"
    val joinTableAlias = "j"
    import CIS._

    val table: SelectBuilder = new SelectBuilder(joinTable)
      .as(joinTableAlias)
      .where(cond.equals(predicateColumn.tableAlias(joinTableAlias), ?))
    val tableOrderByClause = orderByColumn.map(column => AscOrdering(column))

    new JoinBuilder(new SelectBuilder(tableName).as(cisAlias).select(new SqlLiteral(s"$cisAlias.*"))).join(
      table = tableOrderByClause.map(table.orderBy).getOrElse(table),
      cond = cond.equals(ID.tableAlias(cisAlias), joinColumn.tableAlias(joinTableAlias))
    ).query
  }

  def buildSelectCisByIdsQuery(ids: Seq[CiPKType]): String = {
    import CIS._
    new SelectBuilder(tableName).select(new SqlLiteral("*")).where(cond.in(ID, ids)).query
  }
}

trait LookupValuesQueries extends Queries {

  lazy val INSERT_LOOKUP_VALUES: String = {
    import LOOKUP_VALUES._
    sqlb"insert into $tableName ($ci_id, $name, $key, $provider) values (?,?,?,?)"
  }

  lazy val SELECT_LOOKUP_VALUES: String = {
    import LOOKUP_VALUES._
    sqlb"select $name, $key, $provider from $tableName where $ci_id = ?"
  }

  lazy val UPDATE_LOOKUP_VALUE: String = {
    import LOOKUP_VALUES._
    sqlb"update $tableName set $key = ?, $provider = ? where $ci_id = ? and $name = ?"
  }

  lazy val DELETE_LOOKUP_VALUE: String = {
    import LOOKUP_VALUES._
    sqlb"delete from $tableName where $ci_id = ? and $name = ?"
  }

  def buildSelectLookupValuesByCiIdsQuery(ids: Seq[CiPKType]): String = {
    import LOOKUP_VALUES._
    new SelectBuilder(tableName).select(new SqlLiteral("*")).where(cond.in(ci_id, ids)).query
  }
}
