package com.xebialabs.deployit.repository.sql.commands

import com.xebialabs.deployit.core.sql.spring.{MapRowMapper, Setter}
import com.xebialabs.deployit.core.sql.util.{queryWithInClause, queryWithInClausePreserveOrder}
import com.xebialabs.deployit.core.sql.{JoinBuilder, Queries, SchemaInfo, SelectBuilder, asLong, SqlCondition => cond}
import com.xebialabs.deployit.engine.spi.event.CisDeletedEvent
import com.xebialabs.deployit.event.EventBusHolder
import com.xebialabs.deployit.plugin.api.reflect.{DescriptorRegistry, PropertyKind, Type}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.ItemInUseException
import com.xebialabs.deployit.repository.sql.artifacts.ArtifactDataRepository
import com.xebialabs.deployit.repository.sql.base._
import com.xebialabs.deployit.repository.sql.cache.{CisCacheDataProcessor, ConsolidatedCiDeletedInfo}
import com.xebialabs.deployit.repository.sql.properties.{DeleteRepositoryCiProperties, PropertyReader}
import com.xebialabs.deployit.repository.sql.specific.{TypeSpecificDeleter, TypeSpecificReferenceFinder}
import com.xebialabs.deployit.repository.sql.{CiHistoryRepository, CiRepository}
import com.xebialabs.deployit.security.client.PermissionService
import com.xebialabs.deployit.sql.base.schema._
import com.xebialabs.deployit.util.Tuple
import com.xebialabs.license.LicenseCiCounter
import org.springframework.dao.DataIntegrityViolationException
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.sql.ResultSet
import java.util.{ArrayList => JArrayList, List => JList, Map => JMap, Set => JSet}
import scala.annotation.tailrec
import scala.collection.mutable
import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._

class DeleteCommand(override val jdbcTemplate: JdbcTemplate,
                    artifactDataRepository: ArtifactDataRepository,
                    ciRepository: CiRepository,
                    ciHistoryRepository: CiHistoryRepository,
                    licenseCiCounter: LicenseCiCounter,
                    createTypeSpecificDeleters: (Type, CiPKType) => List[TypeSpecificDeleter],
                    createTypeSpecificReferenceFinders: Seq[CiPKType] => List[TypeSpecificReferenceFinder],
                    val ids: Iterable[Tuple[Integer, String]],
                    permissionService: PermissionService,
                    lifecycleHooks: List[CiLifecycleHook],
                    cisCacheDataProcessor: CisCacheDataProcessor)
                   (implicit val schemaInfo: SchemaInfo)
  extends ChangeSetCommand with DeleteCommandQueries with DeleteRepositoryCiProperties with PropertyReader with CiQueries
    with CiExists with SCMTraceabilityDataQueries {

  private val updatedContainmentProperties = new mutable.HashMap[CiPKType, mutable.Set[String]] with mutable.MultiMap[CiPKType, String]

  override def execute(context: ChangeSetContext): Unit = {
    val existingPaths = ciRepository.selectExistingPaths(ids.map(id => idToPath(id.b)).toSet.asJava).asScala
    val extractKey = (map: JMap[String, Object]) => map.get(CIS.path.name).toString
    val existingCis = queryWithInClausePreserveOrder[String, JMap[String, Object]](existingPaths.toSeq, extractKey) { group =>
      jdbcTemplate.query(buildSelectCisByPathsQuery(group), Setter(group), MapRowMapper).asScala
    }
    val cisToDelete: Iterable[(ConfigurationItem, Seq[(CiPKType, String, Type)])] = existingCis.map { map =>
      val path = map.get(CIS.path.name).toString
      val pk = asCiPKType(map.get(CIS.ID.name))
      val ciType = Type.valueOf(map.get(CIS.ci_type.name).asInstanceOf[String])
      map.get(CIS.parent_id.name) match {
        case null => //ignore
        case a =>
          val parentPK = asCiPKType(a)
          deleteAsContainmentValues(path, pk, parentPK)
      }

      if (ciType.toString == "notification.PatSmtpServer") {
        // Check if this PatSmtpServer is referenced in XLD_CONFIGURATIONS
        val normalizedPath = pathToId(path)

        jdbcTemplate.queryForList(SELECT_REFERRING_CONFIGURATION, normalizedPath).asScala.foreach { _ =>
          throw new ItemInUseException(
            "The '%s' configuration cannot be deleted because it is currently used in Personal Access Token (PAT) settings.",
            normalizedPath
          )
        }
      }
      val ci = readBaseCiFromMap(map)
      lifecycleHooks.foreach(_.preDelete(ci))
      deleteTraceabilityData(map.get(CIS.scm_traceability_data_id.name).asInstanceOf[Integer])
      (ci, deletePropertiesAndChildPropertiesReturningPks(pk, path, ciType))
    }

    val cis = cisToDelete.flatMap { case (ci, toDelete) =>
      val deletedCis = ListBuffer[ConfigurationItem]()
      toDelete.foreach { case (pkToDelete, path, ciType) =>
        val id = pathToId(path)
        if (schemaInfo.sqlDialect.abortsTransactionOnException) {
          throwItemInUseIfReferred(pkToDelete, id)
        }
        try {
          context.log("Deleting %s", id)
          deletedCis += getBaseCi(path, ciType, pkToDelete)
          permissionService.removeCiPermissions(pkToDelete)
          ciHistoryRepository.delete(pkToDelete)
          jdbcTemplate.update(DELETE_LOOKUP_VALUES, pkToDelete)
          jdbcTemplate.update(DELETE, pkToDelete)
        } catch {
          case ex: DataIntegrityViolationException =>
            throwItemInUseIfReferred(pkToDelete, id)
            throw ex
        }
        licenseCiCounter.registerTypeRemoval(ciType, context.licenseTransaction)
      }
      deletedCis.toList
    }
    updatedContainmentProperties.foreach { case (pk, properties) =>
      properties.foreach(updateIndexListProperties(pk, _))
    }
    if (cis.nonEmpty) {
      cisCacheDataProcessor.onDelete(ConsolidatedCiDeletedInfo(cis.toList.asJava))
      EventBusHolder.publish(new CisDeletedEvent(cis.toList.asJava))
    }
  }

  private def retrieveCisToDelete(): Map[CiPKType, String] = {
    val (toResolve, resolved) = ids.partition(tuple => Option(tuple.a).isEmpty)
    val result = (queryWithInClause(toResolve.map(id => idToPath(id.b)).toSeq) { group =>
      val query = buildSelectIdsByPathsQuery(group)
      jdbcTemplate.query(query, Setter(group), new RowMapper[(CiPKType, String)] {
        override def mapRow(rs: ResultSet, rowNum: Int): (CiPKType, String) =
          asCiPKType(rs.getInt(CIS.ID.name)) -> rs.getString(CIS.path.name)
      }).asScala
    } ::: resolved.map(tuple => tuple.a -> idToPath(tuple.b)).toList).toSet

    val children = retrieveAllChildren(result, result)

    (result ++ children).toMap
  }

  @tailrec
  private def retrieveAllChildren(cisToDelete: Set[(Integer, String)], toFetchChildren: Set[(Integer, String)]): Set[(Integer, String)] = {
    if (toFetchChildren.isEmpty) {
      cisToDelete
    } else {
      val children = fetchChildren(cisToDelete)
      val combined = cisToDelete ++ children
      retrieveAllChildren(combined, toFetchChildren.diff(combined))
    }
  }

  private def fetchChildren(ids: Set[(Integer, String)]): Set[(Integer, String)] = {
    queryWithInClause(ids.toSeq) { entry =>
      val idCol = CIS.ID
      val pathColumn = CIS.path
      val parentIdCol = CIS.parent_id
      val condition = cond.in(parentIdCol, entry.map(e => e._1))

      val builder = new SelectBuilder(CIS.tableName)
        .distinct()
        .select(idCol)
        .select(pathColumn)
        .where(condition)
      val childIdsQuery = builder.query
      jdbcTemplate.query(childIdsQuery, Setter(builder.parameters), new RowMapper[(CiPKType, String)] {
        override def mapRow(rs: ResultSet, rowNum: Int): (CiPKType, String) =
          asCiPKType(rs.getInt(CIS.ID.name)) -> rs.getString(CIS.path.name)
      }).asScala
    }.toSet
  }

  override def validate(context: ChangeSetContext): Unit = {
    checkReferencesInCache(context.pkCache.keys)
    if (ids.nonEmpty) {
      val toDelete = retrieveCisToDelete()
      val pksToIgnore = context.pkCache.values.toSet

      checkReferencesInGenericProperties(toDelete, pksToIgnore)
      checkReferencesInTypeSpecificTables(toDelete, pksToIgnore)
    }
  }

  /*
   * The (transaction) cache contains CIs already loaded during this action.
   * Check if any of their properties references the CIs we are about to delete.
   * If so, throw an ItemInUseException.
   */
  private def checkReferencesInCache(cache: Iterable[ConfigurationItem]): Unit = {
    val checkedCis = new JArrayList[String]()

    def throwExceptionIfReferenced(ci: ConfigurationItem): Unit = {
      if (!checkedCis.contains(ci.getId)) {
        checkedCis.add(ci.getId)
        ci.getType.getDescriptor.getPropertyDescriptors.asScala.filter(pd =>
          (pd.getKind == PropertyKind.CI ||
            pd.getKind == PropertyKind.SET_OF_CI ||
            pd.getKind == PropertyKind.LIST_OF_CI) && !pd.isAsContainment
        ).foreach {
          def checkItemInUse(pci: ConfigurationItem): Unit = {
            ids.foreach(id =>
              if (pci.getId == id.b)
                throwItemInUseException(id.b, ci.getId)
              else
                throwExceptionIfReferenced(pci)
            )
          }

          _.get(ci) match {
            case pci: ConfigurationItem => checkItemInUse(pci)
            case list: JList[ConfigurationItem@unchecked] => list.asScala.foreach(checkItemInUse)
            case set: JSet[ConfigurationItem@unchecked] => set.asScala.foreach(checkItemInUse)
            case _ => // ignore if null
          }
        }
      }
    }

    cache.foreach(throwExceptionIfReferenced)
  }

  /*
   * Check the database for any generic properties that reference the CIs we are about to delete.
   * CIs that we are about to delete or CIs that are loaded in memory (and already checked) should be ignored.
   * If so, throw an ItemInUseException.
   */
  private def checkReferencesInGenericProperties(cisToDelete: Map[CiPKType, String], pksToIgnore: Set[CiPKType]): Unit = {
    val target = "tar"
    val referrer = "ref"
    val targetID = CIS.ID.tableAlias(target)
    val targetPath = CIS.path.tableAlias(target)
    val targetType = CIS.ci_type.tableAlias(target)
    val referrerID = CI_PROPERTIES.ci_ref_value.tableAlias(referrer)
    val propertyName = CI_PROPERTIES.name.tableAlias(referrer)
    val targetIDAlias = "target_ID"
    val targetPathAlias = "target_path"
    val refererIDAlias = "ref_ID"
    val propertyNameAlias = "property_name"
    val targetTypeAlias = "target_type"
    val referringCis = cisToDelete.keys.grouped(schemaInfo.sqlDialect.inClauseLimit).map { ids =>
      new JoinBuilder(
        new SelectBuilder(CIS.tableName).as(target).distinct().select(targetID, targetIDAlias)
          .select(targetPath, targetPathAlias).select(targetType, targetTypeAlias)
          .select(referrerID, refererIDAlias).select(propertyName, propertyNameAlias)
      ).join(
        new SelectBuilder(CI_PROPERTIES.tableName).as(referrer).distinct().select(CI_PROPERTIES.ci_ref_value)
          .select(CI_PROPERTIES.name).select(CI_PROPERTIES.ci_id)
          .where(cond.in(referrerID, ids)), cond.joinSubselect(cond.equals(CIS.ID.tableAlias(target), CI_PROPERTIES.ci_id.tableAlias(referrer)))
      )
    }.flatMap { builder =>
      jdbcTemplate.query[((CiPKType, String, String, String), (CiPKType, String))](builder.query, Setter(builder.parameters),
        (rs: ResultSet, _: Int) => {
          val pk = asCiPKType(rs.getObject(refererIDAlias))
          (
            (asCiPKType(rs.getObject(targetIDAlias)), rs.getString(targetPathAlias),
              rs.getString(targetTypeAlias), rs.getString(propertyNameAlias)),
            (pk, cisToDelete(pk)),
          )
        }
      ).asScala
    }

    referringCis.find { case ((refPk, _, _, _), (_, _)) =>
      !(cisToDelete.contains(refPk) || pksToIgnore.contains(refPk))
    }.filterNot { case ((_, _, refType, property), (_, _)) =>
      Type.valueOf(refType).getDescriptor.getPropertyDescriptor(property).isAsContainment
    }.foreach { case ((_, refPath, _, _), (_, path)) =>
      throwItemInUseException(pathToId(path), pathToId(refPath))
    }
  }

  private def throwItemInUseIfReferred(pk: CiPKType, id: String): Unit = {
    jdbcTemplate.queryForList(SELECT_REFERRING_CI_PATH, pk).asScala.foreach { map =>
      throwItemInUseException(id, pathToId(map.get(CIS.path.name).asInstanceOf[String]))
    }
    jdbcTemplate.queryForList(SELECT_REFERRING_LOOKUP_VALUE, pk).asScala.foreach { map =>
      throwItemInUseException(id, pathToId(map.get(CIS.path.name).asInstanceOf[String]))
    }
    throwItemInUseIfReferredByTypeSpecificTables(pk, id)
  }

  private def throwItemInUseIfReferredByTypeSpecificTables(pk: CiPKType, id: String): Unit = {
    createTypeSpecificReferenceFinders(Seq(pk)).flatMap(_.findReferences()).foreach { case (refPk: CiPKType, _) =>
      val map = jdbcTemplate.queryForObject(SELECT_CI_BY_ID, MapRowMapper, refPk)
      throwItemInUseException(id, pathToId(map.get(CIS.path.name).asInstanceOf[String]))
    }
  }

  private def checkReferencesInTypeSpecificTables(cisToDelete: Map[CiPKType, String], pksToIgnore: Set[CiPKType]): Unit = {
    val pks = cisToDelete.keys.toSeq
    createTypeSpecificReferenceFinders(pks).flatMap(_.findReferences()).find { case (refPk: CiPKType, _) =>
      !(cisToDelete.contains(refPk) || pksToIgnore.contains(refPk))
    }.map { case (refPk: CiPKType, targetPk: CiPKType) =>
      val map = jdbcTemplate.queryForObject(SELECT_CI_BY_ID, MapRowMapper, refPk)
      throwItemInUseException(pathToId(cisToDelete(targetPk)), pathToId(map.get(CIS.path.name).asInstanceOf[String]))
    }
  }

  private def deleteAsContainmentValues(path: String, pk: CiPKType, parentPk: CiPKType): Unit = {
    jdbcTemplate.queryForList(SELECT_REFERRED_PROPERTIES, parentPk, pk).asScala.map { propertyMap =>
      val propertyName = propertyMap.get(CI_PROPERTIES.name.name).asInstanceOf[String]
      val parentMap = jdbcTemplate.queryForObject(SELECT_CI_BY_ID, MapRowMapper, parentPk)
      val propertyDescriptor = DescriptorRegistry.getDescriptor(Type.valueOf(parentMap.get(CIS.ci_type.name).asInstanceOf[String]))
        .getPropertyDescriptor(propertyName)
      if (!propertyDescriptor.isAsContainment)
        throwItemInUseException(pathToId(path), pathToId(parentMap.get(CIS.path.name).asInstanceOf[String]))
      else {
        deleteProperty(asLong(propertyMap.get(CI_PROPERTIES.ID.name)))
        updatedContainmentProperties.addBinding(parentPk, propertyName)
      }
    }
  }

  private def throwItemInUseException(id: String, refId: String): Nothing = {
    throw new ItemInUseException("Repository entity %s is still referenced by %s", id, refId)
  }

  private def deleteTraceabilityData(traceabilityDataId: Integer) = {
    if (traceabilityDataId != null) {
      jdbcTemplate.update(SCM_TRACEABILITY_DATA_DELETE, traceabilityDataId)
    }
  }

  private def deletePropertiesAndChildPropertiesReturningPks(pk: CiPKType, path: String, ciType: Type): Seq[(CiPKType, String, Type)] = {
    createTypeSpecificDeleters(ciType, pk).foreach(_.deleteProperties())
    jdbcTemplate.update(DELETE_PROPERTIES, pk)
    deleteSourceArtifact(pk)
    deleteChildPropertiesReturningPks(pk) :+ (pk, path, ciType)
  }

  private def deleteSourceArtifact(pk: CiPKType): Unit = {
    jdbcTemplate.update(DELETE_SOURCE_ARTIFACTS, pk)
    artifactDataRepository.delete(pk)
  }

  private def deleteChildPropertiesReturningPks(pk: CiPKType): Seq[(CiPKType, String, Type)] = {
    jdbcTemplate.query(SELECT_CI_BY_PARENT_ID, MapRowMapper, pk).asScala.flatMap { map =>
      val pk = asCiPKType(map.get(CIS.ID.name))
      val ciType = Type.valueOf(map.get(CIS.ci_type.name).asInstanceOf[String])
      val path = map.get(CIS.path.name).asInstanceOf[String]
      deleteTraceabilityData(map.get(CIS.scm_traceability_data_id.name).asInstanceOf[Integer])
      deletePropertiesAndChildPropertiesReturningPks(pk, path, ciType)
    }.toSeq
  }

  private def updateIndexListProperties(pk: CiPKType, propertyName: String): Unit = {
    // prepared statement
    var idx = 0
    jdbcTemplate.queryForList(SELECT_PROPERTIES_BY_NAME, pk, propertyName).asScala.sortBy(getIndexValue).foreach { map =>
      val propPk = asCiPKType(map.get(CI_PROPERTIES.ID.name))
      jdbcTemplate.update(UPDATE_PROPERTY_IDX, idx.asInstanceOf[Integer], propPk)
      idx += 1
    }
  }
}

trait DeleteCommandQueries extends Queries {

  lazy val DELETE: String = {
    import CIS._
    sqlb"delete from $tableName where $ID = ?"
  }

  lazy val SELECT_REFERRING_CI_PATH: String = {
    import com.xebialabs.deployit.sql.base.schema.CI_PROPERTIES
    sqlb"select ${CIS.path} from ${CIS.tableName} ci inner join ${CI_PROPERTIES.tableName} prop on ci.${CIS.ID} = prop.${CI_PROPERTIES.ci_id} where prop.${CI_PROPERTIES.ci_ref_value} = ?"
  }

  lazy val SELECT_REFERRED_PROPERTIES: String = {
    import CI_PROPERTIES._
    sqlb"select * from $tableName where $ci_id = ? and $ci_ref_value = ?"
  }

  lazy val DELETE_PROPERTIES: String = {
    import CI_PROPERTIES._
    sqlb"delete from $tableName where $ci_id = ?"
  }

  lazy val SELECT_PROPERTIES_BY_NAME: String = {
    import CI_PROPERTIES._
    sqlb"select * from $tableName where $ci_id = ? and $name = ?"
  }

  lazy val UPDATE_PROPERTY_IDX: String = {
    import CI_PROPERTIES._
    sqlb"update $tableName set $idx = ? where $ID = ?"
  }

  lazy val SELECT_CI_TREE_BY_PATH: String = {
    import CIS._
    sqlb"select $ID, $path from $tableName where $path = ? or $path like ? order by $path"
  }

  lazy val SELECT_REFERRING_LOOKUP_VALUE: String = {
    sqlb"select ${CIS.path} from ${CIS.tableName} c inner join ${LOOKUP_VALUES.tableName} l on c.${CIS.ID} = l.${LOOKUP_VALUES.ci_id} where ${LOOKUP_VALUES.provider} = ?"
  }

  lazy val DELETE_LOOKUP_VALUES: String = {
    import LOOKUP_VALUES._
    sqlb"delete from $tableName where $ci_id = ?"
  }

  lazy val DELETE_SOURCE_ARTIFACTS: String = {
    import com.xebialabs.deployit.repository.sql.persisters.SourceArtifactSchema._
    sqlb"delete from $tableName where $ID = ?"
  }

  lazy val SELECT_REFERRING_CONFIGURATION: String = {
    import com.xebialabs.deployit.repository.sql.settings.ConfigurationSchema
    // Join the configuration table to itself to check both conditions
    sqlb"""
      select * from ${ConfigurationSchema.tableName} smtp_conf
      inner join ${ConfigurationSchema.tableName} enabled_conf
        on smtp_conf.${ConfigurationSchema.key} = 'deploy.configuration.setting.pat.pat_email_notification_smtp_server_ci_name'
        and enabled_conf.${ConfigurationSchema.key} = 'deploy.configuration.setting.pat.pat_email_notification_enabled'
        and smtp_conf.${ConfigurationSchema.value} = ?
        and enabled_conf.${ConfigurationSchema.value} = 'true'
    """
  }
}
