package com.xebialabs.deployit.repository.sql.specific.configurable

import com.xebialabs.deployit.plugin.api.reflect.PropertyKind._
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.sql.base.{CiPKType, asCiPKType}
import com.xebialabs.deployit.repository.sql.reader.properties.CiDataProvider
import com.xebialabs.deployit.repository.sql.specific.columns.PropertyColumnsReader
import com.xebialabs.deployit.repository.sql.specific.configurable.ConfigurableTypeSpecificPersisterFactory.CiFromMapReader
import com.xebialabs.deployit.repository.sql.specific.{TypeSpecificPersisterFactory, TypeSpecificReader}
import grizzled.slf4j.Logging
import org.springframework.dao.EmptyResultDataAccessException

import java.util.{List => JList}
import scala.util.Try

class ConfigurableReader(val pk: CiPKType,
                         val columns: Option[PropertyColumnsReader],
                         val tables: Map[String, PropertyTable[_, _]])
  extends TypeSpecificReader with Logging {

  override def readProperties(ci: ConfigurationItem,
                              ciResolver: CiPKType => ConfigurationItem,
                              convert: (PropertyDescriptor, Any) => Any)
                             (implicit ciDataProvider: CiDataProvider,
                              readFromMap: CiFromMapReader,
                              typeSpecificPersisterFactories: JList[TypeSpecificPersisterFactory]): Unit = {
    val descriptor = ci.getType.getDescriptor

    def resolveColumnCis(kind: PropertyKind, value: Any): Any = kind match {
      case CI if value != null => ciResolver(asCiPKType(value))
      case SET_OF_CI | LIST_OF_CI => throw new IllegalStateException(s"Invalid kind $kind for PropertyColumnsReader")
      case _ => value
    }

    def resolveRelationsTableCis(kind: PropertyKind, propertyTable: PropertyTable[_, _]): Any = kind match {
      case CI => Option(propertyTable.reader.readProperty(pk)).map(ciId => ciResolver(asCiPKType(ciId))).orNull
      case _ => propertyTable.reader.readProperty(pk)
    }

    def readColumnProperties(): Unit = {
      columns.map { c =>
        Try(c.readProperties(pk))
          .map {
            _.foreach { case (propertyName, value) =>
              val propertyDescriptor = descriptor.getPropertyDescriptor(propertyName)
              if (propertyDescriptor != null)
                propertyDescriptor.set(ci, convert(propertyDescriptor, resolveColumnCis(propertyDescriptor.getKind, value)))
            }
          }
          .recover {
            case _: EmptyResultDataAccessException =>
              logger.warn(s"Couldn't read properties for $ci from table '${c.table.name}' (pk: $pk). This is probably corrupted data, which can be repaired by editing the CI.")
          }
      }
    }

    def readTableProperties(): Unit = {
      tables.foreach { case (propertyName, propertyTable) =>
        val propertyDescriptor = descriptor.getPropertyDescriptor(propertyName)
        val value = resolveRelationsTableCis(propertyDescriptor.getKind, propertyTable)
        propertyDescriptor.set(ci, convert(propertyDescriptor, value))
      }
    }

    readColumnProperties()
    readTableProperties()
  }
}
