package com.xebialabs.deployit.repository.sql.specific.tables

import com.xebialabs.deployit.core.sql.batch.BatchCommandWithSetter
import com.xebialabs.deployit.core.sql.spring.Setter
import com.xebialabs.deployit.core.sql.{ColumnName, SchemaInfo, TableName}
import com.xebialabs.deployit.core.util.CollectionUtil
import com.xebialabs.deployit.repository.sql.base._
import com.xebialabs.deployit.repository.sql.reader.properties.CiDataProvider
import com.xebialabs.deployit.repository.sql.specific.TypeSpecificPersisterFactory
import com.xebialabs.deployit.repository.sql.specific.configurable.ConfigurableTypeSpecificPersisterFactory.CiFromMapReader
import com.xebialabs.deployit.repository.sql.specific.configurable.{PropertyTableInserter, PropertyTableReader, PropertyTableSelectDescriptor, PropertyTableUpdater}
import org.springframework.jdbc.core.JdbcTemplate

import java.util.{ArrayList => JArrayList, List => JList, Map => JMap}
import scala.annotation.tailrec
import scala.jdk.CollectionConverters._

class MapStringStringPropertyTable(val table: TableName, val idColumn: ColumnName, val keyColumn: ColumnName, val valueColumn: ColumnName)
                                  (implicit val jdbcTemplate: JdbcTemplate, implicit val schemaInfo: SchemaInfo)
  extends MultiValuePropertyTable[JMap[String, String], JMap[String, String]]
    with StringCollectionPropertyTable[JMap[String, String], JMap[String, String]] {

  override implicit val selectDescriptor: PropertyTableSelectDescriptor = PropertyTableSelectDescriptor(
    table, List(keyColumn, valueColumn), idColumn
  )

  val INSERT = sqlb"insert into $table ($idColumn, $keyColumn, $valueColumn) values (?, ?, ?)"
  val UPDATE = sqlb"update $table set $valueColumn = ? where $idColumn = ? and $keyColumn = ?"
  val DELETE = sqlb"delete from $table where $idColumn = ? and $keyColumn = ?"

  private def retrieveValueMap(pk: CiPKType)(implicit ciDataProvider: CiDataProvider): JMap[String, String] =
    CollectionUtil.toMap(ciDataProvider.getPropertyTableValues(pk, selectDescriptor), keyColumn.name, valueColumn.name)

  override val reader: PropertyTableReader[JMap[String, String]] = new PropertyTableReader[JMap[String, String]] {

    override def readProperty(pk: CiPKType)
                             (implicit ciDataProvider: CiDataProvider,
                              readFromMap: CiFromMapReader,
                              typeSpecificPersisterFactories: JList[TypeSpecificPersisterFactory]): JMap[String, String] =
      retrieveValueMap(pk)
  }

  override val inserter: PropertyTableInserter[JMap[String, String]] = new PropertyTableInserter[JMap[String, String]] {

    override def insertTypedProperty(pk: CiPKType, value: JMap[String, String], handler: ErrorHandler): Unit = {
      val list = new JArrayList(value.entrySet())
      for (i <- 0 until value.size()) {
        handler {
          val entry = list.get(i)
          jdbcTemplate.update(INSERT, Setter(Seq(pk, entry.getKey, entry.getValue)))
        }
      }
    }

    override def batchInsertTypedProperty(pk: CiPKType, value: JMap[String, String]): List[BatchCommandWithSetter] = {
      value.entrySet.asScala.toList.map { entry =>
        BatchCommandWithSetter(INSERT, Setter(Seq(pk, entry.getKey, entry.getValue)))
      }
    }
  }

  override val updater: PropertyTableUpdater[JMap[String, String]] = new PropertyTableUpdater[JMap[String, String]] {
    override def updateTypedProperty(pk: CiPKType, newValues: JMap[String, String])
                                    (implicit ciDataProvider: CiDataProvider): Unit = {
      @tailrec
      def applyChanges(oldValues: List[(String, String)],
                       newValues: Map[String, String]): Unit = oldValues match {
        case (key, oldValue) :: xs => newValues.get(key) match {
          case Some(newValue) if newValue != oldValue =>
            jdbcTemplate.update(UPDATE, Setter(Seq(newValue, pk, key)))
            applyChanges(xs, newValues - key)
          case None =>
            jdbcTemplate.update(DELETE, Setter(Seq(pk, key)))
            applyChanges(xs, newValues)
          case _ => applyChanges(xs, newValues - key)
        }
        case Nil => newValues.foreach { case (key, newValue) =>
          jdbcTemplate.update(INSERT, Setter(Seq(pk, key, newValue)))
        }
      }
      applyChanges(retrieveValueMap(pk).asScala.toList, newValues.asScala.toMap)
    }

    override def batchUpdateTypedProperty(pk: CiPKType, newValues: JMap[String, String])
                                         (implicit ciDataProvider: CiDataProvider): List[BatchCommandWithSetter] = {
      @tailrec
      def buildChanges(oldValues: List[(String, String)],
                       newValues: Map[String, String],
                       acc: Vector[BatchCommandWithSetter] = Vector()): List[BatchCommandWithSetter] = oldValues match {
        case (key, oldValue) :: xs => newValues.get(key) match {
          case Some(newValue) if newValue != oldValue =>
            buildChanges(xs, newValues - key, acc :+ BatchCommandWithSetter(UPDATE, Setter(Seq(newValue, pk, key))))
          case None => buildChanges(xs, newValues, acc :+ BatchCommandWithSetter(DELETE, Setter(Seq(pk, key))))
          case _ => buildChanges(xs, newValues - key, acc)
        }
        case Nil => acc.toList ::: newValues.toList.map { case (key, newValue) =>
          BatchCommandWithSetter(INSERT, Setter(Seq(pk, key, newValue)))
        }
      }
      buildChanges(retrieveValueMap(pk).asScala.toList, newValues.asScala.toMap)
    }
  }
}
