package com.xebialabs.deployit.artifact.resolution

import java.io.InputStream
import java.net.{URI, URLEncoder}
import java.util.concurrent.atomic.AtomicReference
import com.xebialabs.deployit.artifact.resolution.InternalArtifactResolver._
import com.xebialabs.deployit.engine.spi.artifact.resolution.ArtifactResolver.Resolver
import com.xebialabs.deployit.engine.spi.artifact.resolution.{ArtifactResolver, ResolvedArtifactFile}
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact
import com.xebialabs.deployit.repository.sql.artifacts.ArtifactDataRepository
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.Lazy
import org.slf4j.LoggerFactory

import scala.util.Try

@Resolver(protocols = Array(Protocol))
class InternalArtifactResolver extends ArtifactResolver {
  private val logger = LoggerFactory.getLogger(classOf[InternalArtifactResolver])
  override def validateCorrectness(artifact: SourceArtifact): Boolean = canHandle(artifact.getFileUri)

  override def resolveLocation(artifact: SourceArtifact): ResolvedArtifactFile = new ResolvedArtifactFile {
    override def openStream(): InputStream = holder.get().retrieve(artifact.getId)
    override def getFileName: String = URI.create(URLEncoder.encode(artifact.getFileUri, "UTF-8")).getPath
    override def getFileSize: Long = {
      logger.debug(s"stream:2G:Retrieving file size for artifact: ${artifact.getId}")
      try {
        Option(holder.get()).map { repo =>
          logger.debug(s"stream:2G:Repository type: ${repo.getClass.getName} for artifact: ${artifact.getId}")
          val size = repo.retrieveFileSize(artifact.getId)
          logger.debug(s"stream:2G:InternalArtifactResolver.getFileSize: Retrieved size $size bytes for artifact ${artifact.getId}")
          size
        }.getOrElse {
          logger.debug(s"stream:2G:Repository is NULL for artifact: ${artifact.getId}")
          -1L
        }
      } catch {
        case e: Exception =>
          logger.warn(s"stream:2G:Error retrieving file size for artifact ${artifact.getId}: ${e.getMessage}", e)
          -1L
      }
    }
    override def close(): Unit = {}
  }
}

object InternalArtifactResolver {

  final val Protocol = "internal"

  def canHandle(uri: String): Boolean = Try(URI.create(uri)).map(_.getScheme == Protocol).getOrElse(false)

  private final val holder = new AtomicReference[ArtifactDataRepository]()

  @Lazy
  @Autowired
  def setArtifactRepository(artifactDataRepository: ArtifactDataRepository): Unit = {
    holder.getAndSet(artifactDataRepository)
  }

  def getInstance(): InternalArtifactResolver.type = this
}
