package com.xebialabs.deployit.repository.sql.properties

import java.util
import com.xebialabs.deployit.core.{ListOfStringView, MapStringStringView, SetOfStringView, StringValue}
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.StringValueConverter.valueToString
import com.xebialabs.deployit.sql.base.schema._
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.deployit.core.util.TypeConversions._
import com.xebialabs.deployit.core.util.CollectionUtil
import com.xebialabs.deployit.repository.sql.base._

trait PropertyReader {

  import CI_PROPERTIES._
  import PropertyKind._

  protected def getSimpleValue(k: PropertyKind, map: util.Map[String, AnyRef]): Any =
    k match {
      case BOOLEAN => getBooleanValue(map).getOrElse(getStringValue(map))
      case ENUM | STRING | LIST_OF_STRING | SET_OF_STRING | MAP_STRING_STRING => getStringValue(map)
      case INTEGER => getIntegerValue(map).getOrElse(getStringValue(map))
      case DATE => getDateValue(map).getOrElse(getStringValue(map))
      case CI | LIST_OF_CI | SET_OF_CI => getCiRefValue(map)
    }

  private def getBooleanValue(map: util.Map[String, AnyRef]) =
    Option(map.get(boolean_value.name)).map { enforceBoolean }

  private def getIntegerValue(map: util.Map[String, AnyRef]) =
    Option(map.get(integer_value.name)).map { enforceInteger }

  protected def getStringValue(map: util.Map[String, AnyRef]): String =
    map.get(string_value.name).asInstanceOf[String]

  private def getDateValue(map: util.Map[String, AnyRef]) =
    Option(map.get(date_value.name)).map { enforceDate }

  protected def getCiRefValue(map: util.Map[String, AnyRef]): CiPKType =
    asCiPKType(map.get(ci_ref_value.name))

  protected def getIndexValue(map: util.Map[String, AnyRef]): Int =
    map.get(idx.name).asInstanceOf[Number].intValue()

  protected def getKeyValue(map: util.Map[String, AnyRef]): String =
    map.get(key.name).asInstanceOf[String]

}

trait EncryptedPropertyReader extends PropertyReader {

  val passwordEncrypter: PasswordEncrypter

  protected def getStringValue(map: util.Map[String, AnyRef], isPassword: Boolean): String = {
    var value = getStringValue(map)
    if (isPassword) value = passwordEncrypter.ensureDecrypted(value)
    value
  }

}

trait PropertyManipulator extends PropertyReader {

  val passwordEncrypter: PasswordEncrypter

  protected def getValue(item: ConfigurationItem, pd: PropertyDescriptor, findPkFor: ConfigurationItem => CiPKType): Option[Any] = Option(pd.get(item)).map { value: AnyRef =>
    import scala.compat.java8.FunctionConverters._
    pd.getKind match {
      case PropertyKind.STRING if pd.isPassword =>
        passwordEncrypter.ensureEncrypted(value.toString)
      case PropertyKind.SET_OF_STRING =>
        var set = value match {
          case set: SetOfStringView => set
          case _ => SetOfStringView.from(value.asInstanceOf[util.Set[String]])
        }
        if (pd.isPassword)
          set = set.encrypt
        CollectionUtil.apply[StringValue, String](set.getWrapped, valueToString(passwordEncrypter))
      case PropertyKind.LIST_OF_STRING =>
        var list = value match {
          case list: ListOfStringView => list
          case _ => ListOfStringView.from(value.asInstanceOf[util.List[String]])
        }
        if (pd.isPassword)
          list = list.encrypt
        CollectionUtil.apply[StringValue, String](list.getWrapped, valueToString(passwordEncrypter))
      case PropertyKind.MAP_STRING_STRING =>
        var map = value match {
          case map: MapStringStringView => map
          case _ => MapStringStringView.from(value.asInstanceOf[util.Map[String, String]])
        }
        if (pd.isPassword)
          map = map.encrypt
        CollectionUtil.apply[StringValue, String](map.getWrapped, valueToString(passwordEncrypter))
      case PropertyKind.ENUM if value.isInstanceOf[String] => value
      case PropertyKind.ENUM if value.isInstanceOf[Enum[_]] => value.asInstanceOf[Enum[_]].name()
      case PropertyKind.CI => findPkFor(value.asInstanceOf[ConfigurationItem])
      case PropertyKind.SET_OF_CI => CollectionUtil.apply[ConfigurationItem, CiPKType](value.asInstanceOf[util.Set[ConfigurationItem]], findPkFor.asJava)
      case PropertyKind.LIST_OF_CI => CollectionUtil.apply[ConfigurationItem, CiPKType](value.asInstanceOf[util.List[ConfigurationItem]], findPkFor.asJava)
      case _ => value
    }
  }

}
