package com.xebialabs.deployit.repository.sql.cache

import ai.digital.deploy.cache.service.CacheDataService
import com.xebialabs.deployit.repository.sql.base.{CiPKType, asCiPKType, idToPath}
import com.xebialabs.deployit.sql.base.schema.CIS
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service

import java.util
import java.util.{List => JList, Map => JMap}
import scala.collection.mutable
import scala.jdk.CollectionConverters._

@Service
class CiCacheDataServiceFacade(ciPkCacheDataService: CacheDataService[CiPKType, JMap[String, Object]],
                               ciPathCacheDataService: CacheDataService[String, JMap[String, Object]],
                               ciPropertiesCacheDataService: CacheDataService[CiPKType, JList[JMap[String, Object]]])
  extends Logging {

  def get(pk: CiPKType): Option[JMap[String, Object]] = {
    ciPkCacheDataService.get(pk)
  }

  def getWithFallback(pk: CiPKType, fallback: CiPKType => JMap[String, Object]): JMap[String, Object] = {
    ciPkCacheDataService.getWithFallback(pk, key => {
      val result = fallback(key)
      put(result)
      result
    })
  }

  def getAllPksWithFallback(pks: Seq[CiPKType], fallback: Seq[CiPKType] => List[JMap[String, Object]]): List[JMap[String, Object]] = {
    val found = ciPkCacheDataService.getAll(pks.toSet.asJava).getOrElse(new util.HashMap())
    val notFound: Set[CiPKType] = pks.toSet.diff(found.keySet().asScala)
    val cis = found.values().asScala.toList
    logger.trace(s"getAllPksWithFallback(pks): pks($pks) || found($found) || notFound($notFound)")
    if (notFound.nonEmpty) {
      val result = fallback(notFound.toSeq)
      put(result)
      cis ++ result
    } else {
      cis
    }
  }

  def getAllByPk: Option[JMap[CiPKType, JMap[String, Object]]] = {
    ciPkCacheDataService.getAll
  }

  def get(id: String): Option[JMap[String, Object]] = {
    ciPathCacheDataService.get(idToPath(id))
  }

  def getWithFallback(id: String, fallback: String => JMap[String, Object]): JMap[String, Object] = {
    val path = idToPath(id)
    get(path).getOrElse({
      val result = fallback(path)
      put(result)
      result
    })
  }

  def getAllPathsWithFallback(ids: Seq[String], fallback: Seq[String] => mutable.Buffer[JMap[String, Object]]): List[JMap[String, Object]] = {
    val paths = ids.map(idToPath)
    val found = ciPathCacheDataService.getAll(paths.toSet.asJava).getOrElse(new util.HashMap())
    val notFound: Set[String] = paths.toSet.diff(found.keySet().asScala)
    val cis = found.values().asScala.toList
    logger.trace(s"getAllPathsWithFallback(ids): pks($ids) || found($found) || notFound($notFound)")
    if (notFound.nonEmpty) {
      val result = fallback(notFound.toSeq)
      put(result.toList)
      cis ++ result
    } else {
      cis
    }
  }

  def getAllByPath: Option[JMap[String, JMap[String, Object]]] = {
    ciPathCacheDataService.getAll
  }

  def clearAll(): Unit = {
    ciPkCacheDataService.clear()
    ciPathCacheDataService.clear()
    ciPropertiesCacheDataService.clear()
  }

  def put(value: JMap[String, Object]): Unit = {
    if (value != null) {
      val pk = asCiPKType(value.get(CIS.ID.name))
      val path = value.get(CIS.path.name).toString
      ciPkCacheDataService.put(pk, value)
      ciPathCacheDataService.put(path, value)
    }
  }

  def put(cis: List[JMap[String, Object]]): Unit = {
    if (cis.nonEmpty) {
      cis.foreach(
        map => {
          val pk = asCiPKType(map.get(CIS.ID.name))
          val path = map.get(CIS.path.name).toString
          ciPkCacheDataService.put(pk, map)
          ciPathCacheDataService.put(path, map)
        }
      )
    }
  }

  def remove(pkIdPair: (CiPKType, String)): Unit = {
    ciPkCacheDataService.remove(pkIdPair._1)
    ciPropertiesCacheDataService.remove(pkIdPair._1)
    if (pkIdPair._2 != null) {
      val path = idToPath(pkIdPair._2)
      ciPathCacheDataService.remove(path)
    }
    logger.trace(s"remove: entries in ${ciPkCacheDataService.cacheName} ::: ${ciPkCacheDataService.getAll}")
    logger.trace(s"remove: entries in ${ciPathCacheDataService.cacheName} ::: ${ciPathCacheDataService.getAll}")
    logger.trace(s"remove: entries in ${ciPropertiesCacheDataService.cacheName} ::: ${ciPropertiesCacheDataService.getAll}")
  }

  def removeAll(pkIdPairs: Map[CiPKType, String]): Unit = {
    val pks = pkIdPairs.keySet.asJava
    val paths = pkIdPairs.values.filterNot(_ == null).map(idToPath).toSet.asJava
    ciPkCacheDataService.removeAll(pks)
    ciPathCacheDataService.removeAll(paths)
    ciPropertiesCacheDataService.removeAll(pks)
    logger.trace(s"removeAll: entries in ${ciPkCacheDataService.cacheName} ::: ${ciPkCacheDataService.getAll}")
    logger.trace(s"removeAll: entries in ${ciPathCacheDataService.cacheName} ::: ${ciPathCacheDataService.getAll}")
    logger.trace(s"removeAll: entries in ${ciPropertiesCacheDataService.cacheName} ::: ${ciPropertiesCacheDataService.getAll}")
  }

  def removeAllProperties(pkIds: Set[CiPKType]) : Unit = {
    ciPropertiesCacheDataService.removeAll(pkIds.filterNot(_ == null).asJava)
  }

  def getPropertiesByPkWithFallback(pk: CiPKType, fallback: CiPKType => JList[JMap[String, Object]]): JList[JMap[String, Object]] = {
    getPropertiesByPk(pk).getOrElse({
      val value = fallback(pk)
      if (value != null && !value.isEmpty) putProperties(pk, value)
      value
    })
  }

  def getAllProperties: Option[JMap[CiPKType, JList[JMap[String, Object]]]] = {
    ciPropertiesCacheDataService.getAll
  }

  def getPropertiesByPk(pk: CiPKType): Option[JList[JMap[String, Object]]] = {
    ciPropertiesCacheDataService.get(pk)
  }

  def putProperties(pk: CiPKType, value: JList[JMap[String, Object]]): Unit = {
    if (value != null && !value.isEmpty) {
      ciPropertiesCacheDataService.put(pk, value)
    }
  }

  def getAllPropertiesWithFallback(ids: Seq[CiPKType], fallback: Seq[CiPKType] => Map[CiPKType, List[JMap[String, Object]]]): JMap[CiPKType, JList[JMap[String, Object]]] = {
    val found = ciPropertiesCacheDataService.getAll(ids.toSet.asJava).getOrElse(new util.HashMap())
    val notFound: Set[CiPKType] = ids.toSet.diff(found.keySet().asScala)
    val cis = new util.HashMap[CiPKType, JList[JMap[String, Object]]](found)
    logger.trace(s"getAllPropertiesWithFallback(ids): pks($ids) || found($found) || notFound($notFound)")
    if (notFound.nonEmpty) {
      val result = fallback(notFound.toSeq)
      result.foreach { case (key, value) =>
        val convertedValue = new util.ArrayList[JMap[String, Object]](value.asJava)
        putProperties(key, convertedValue)
        cis.put(key, convertedValue)
      }
      cis
    } else {
      cis
    }
  }

}
