package com.xebialabs.deployit.repository.sql.specific.tables

import com.xebialabs.deployit.core.sql._
import com.xebialabs.deployit.core.sql.batch.BatchCommandWithSetter
import com.xebialabs.deployit.core.sql.spring.Setter
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.sql.base._
import com.xebialabs.deployit.repository.sql.reader.properties.{CiDataProvider, CiGroupCiDataProvider}
import com.xebialabs.deployit.repository.sql.specific.TypeSpecificPersisterFactory
import com.xebialabs.deployit.repository.sql.specific.configurable.ConfigurableTypeSpecificPersisterFactory.CiFromMapReader
import com.xebialabs.deployit.repository.sql.specific.configurable.{PropertyTableInserter, PropertyTableReader, PropertyTableSelectDescriptor, PropertyTableUpdater}
import org.springframework.jdbc.core.JdbcTemplate

import java.util.stream.Collectors
import java.util.{List => JList, Set => JSet}
import scala.annotation.tailrec
import scala.jdk.CollectionConverters._

abstract class SetPropertyTable[T, V](val ct: Class[T])
  extends MultiValuePropertyTable[JSet[T], JSet[V]] with Queries {
  val valueColumn: ColumnName

  override implicit val selectDescriptor: PropertyTableSelectDescriptor = PropertyTableSelectDescriptor(
    table, List(valueColumn), idColumn, Nil
  )

  val INSERT = sqlb"insert into $table ($idColumn, $valueColumn) values (?, ?)"
  val DELETE = sqlb"delete from $table where $idColumn = ? and $valueColumn = ?"

  def toParam(value: T): Any

  protected def ensureT(value: Any): T = value match {
    case matches: T => matches
    case _ => throw new IllegalArgumentException(s"Value $value doesn't match requested type ${ct.getCanonicalName}")
  }

  private def retrieveValueSet[X](pk: CiPKType)(implicit ciDataProvider: CiDataProvider): JSet[X] =
    ciDataProvider
      .getPropertyTableValues(pk, selectDescriptor)
      .asScala
      .map(map => ensureT(map.get(valueColumn.name)).asInstanceOf[X])
      .toSet
      .asJava

  override val reader: PropertyTableReader[JSet[V]] = new PropertyTableReader[JSet[V]] {
    override def readProperty(pk: CiPKType)
                             (implicit ciDataProvider: CiDataProvider,
                              readFromMap: CiFromMapReader,
                              typeSpecificPersisterFactories: JList[TypeSpecificPersisterFactory]): JSet[V] =
      retrieveValueSet(pk)
  }

  override val inserter: PropertyTableInserter[JSet[T]] = new PropertyTableInserter[JSet[T]] {
    override def insertTypedProperty(pk: CiPKType, value: JSet[T], handler: ErrorHandler): Unit = value.forEach { v =>
      handler {
        jdbcTemplate.update(INSERT, Setter(Seq(pk, toParam(v))))
      }
    }

    override def batchInsertTypedProperty(pk: CiPKType, value: JSet[T]): List[BatchCommandWithSetter] =
      value.asScala.toList.map(v => BatchCommandWithSetter(INSERT, Setter(Seq(pk, toParam(v)))))
  }

  override val updater: PropertyTableUpdater[JSet[T]] = new PropertyTableUpdater[JSet[T]] {
    override def updateTypedProperty(pk: CiPKType, newValues: JSet[T])(implicit ciDataProvider: CiDataProvider): Unit = {
      @tailrec
      def applyChanges(oldValues: List[T],
                       newValues: Set[T]): Unit = oldValues match {
        case oldValue :: xs if !newValues.contains(oldValue) =>
          jdbcTemplate.update(DELETE, Setter(Seq(pk, oldValue)))
          applyChanges(xs, newValues)
        case newValue :: xs => applyChanges(xs, newValues - newValue)
        case Nil => newValues.foreach(newValue => jdbcTemplate.update(INSERT, Setter(Seq(pk, newValue))))
      }

      applyChanges(retrieveValueSet(pk).asScala.toList, newValues.asScala.toSet)
    }

    override def batchUpdateTypedProperty(pk: CiPKType, newValues: JSet[T])
                                         (implicit ciDataProvider: CiDataProvider): List[BatchCommandWithSetter] = {
      @tailrec
      def buildChanges(oldValues: List[T],
                       newValues: Set[T],
                       acc: Vector[BatchCommandWithSetter] = Vector()): List[BatchCommandWithSetter] = oldValues match {
        case oldValue :: xs if !newValues.contains(oldValue) =>
          buildChanges(xs, newValues, acc :+ BatchCommandWithSetter(DELETE, Setter(Seq(pk, oldValue))))
        case oldValue :: xs => buildChanges(xs, newValues - oldValue, acc)
        case Nil => acc.toList ::: newValues.toList.map(newValue => BatchCommandWithSetter(INSERT, Setter(Seq(pk, newValue))))
      }

      buildChanges(retrieveValueSet(pk).asScala.toList, newValues.asScala.toSet)
    }
  }
}

class SetCiPropertyTable(val table: TableName, val idColumn: ColumnName, val valueColumn: ColumnName)
                        (implicit val jdbcTemplate: JdbcTemplate,
                         implicit val schemaInfo: SchemaInfo)
  extends SetPropertyTable[CiPKType, ConfigurationItem](classOf[CiPKType])
    with CiCollectionPropertyTable[JSet[CiPKType], JSet[ConfigurationItem]] {

  override implicit val selectDescriptor: PropertyTableSelectDescriptor = PropertyTableSelectDescriptor(
    table, List(valueColumn), idColumn, Nil, ciRefColumn = Some(valueColumn)
  )

  override def toParam(value: CiPKType): Any = asCiPKType(value)

  override protected def ensureT(value: Any): CiPKType = asCiPKType(value)

  override val reader: PropertyTableReader[JSet[ConfigurationItem]] = new PropertyTableReader[JSet[ConfigurationItem]] {
    override def readProperty(pk: CiPKType)
                             (implicit ciDataProvider: CiDataProvider,
                              readFromMap: CiFromMapReader,
                              typeSpecificPersisterFactories: JList[TypeSpecificPersisterFactory]): JSet[ConfigurationItem] = {
      val (lists, ciPkAndTypes) = retrieveMaps(pk)
      val setDataProvider = new CiGroupCiDataProvider(ciPkAndTypes)
      lists.stream().map[ConfigurationItem](readFromMap(_, setDataProvider)).collect(Collectors.toSet())
    }
  }
}

class SetStringPropertyTable(val table: TableName, val idColumn: ColumnName, val valueColumn: ColumnName)
                            (implicit val jdbcTemplate: JdbcTemplate, implicit val schemaInfo: SchemaInfo)
  extends SetPropertyTable[String, String](classOf[String]) with StringCollectionPropertyTable[JSet[String], JSet[String]] {
  override def toParam(value: String): Any = value
}
