package com.xebialabs.deployit.repository.sql.coc

import java.sql.{Connection, PreparedStatement}

import com.xebialabs.deployit.core.sql.spring.{MapRowMapper, Setter}
import com.xebialabs.deployit.core.sql.{SchemaInfo, setTimestamp}
import com.xebialabs.deployit.repository.sql.base.SCMTraceabilityDataQueries
import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData
import com.xebialabs.xlplatform.coc.service.SCMTraceabilityService
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.jdbc.support.GeneratedKeyHolder
import org.springframework.stereotype.Service
import org.springframework.transaction.annotation.Transactional

import scala.jdk.CollectionConverters._

@Service
@Transactional("mainTransactionManager")
class DefaultSCMTraceability(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate)
                            (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends SCMTraceabilityService with SCMTraceabilityDataQueries {
  private def fillInTraceabilityParams(ps: PreparedStatement, scmData: SCMTraceabilityData): PreparedStatement = {
    ps.setString(1, scmData.getKind)
    ps.setString(2, scmData.getCommit)
    ps.setString(3, scmData.getAuthor)
    setTimestamp(ps, 4, scmData.getDate)
    Setter.setString(ps, 5, scmData.getMessage)
    ps.setString(6, scmData.getRemote)
    ps.setString(7, scmData.getFileName)
    ps
  }

  override def getById(id: Integer): Option[SCMTraceabilityData] = {
    import com.xebialabs.deployit.repository.sql.base.SCMTraceabilityDBData._
    jdbcTemplate
      .query(SCM_TRACEABILITY_DATA_SELECT_BY_ID, MapRowMapper, id)
      .asScala
      .headOption
      .map { map =>
        new SCMTraceabilityData(
          map.get(kind.name).asInstanceOf[String],
          map.get(commit.name).asInstanceOf[String],
          map.get(author.name).asInstanceOf[String],
          map.get(date.name).asInstanceOf[DateTime],
          map.get(message.name).asInstanceOf[String],
          map.get(remote.name).asInstanceOf[String],
          map.get(file_name.name).asInstanceOf[String]
        )
      }
  }

  override def insert(scmData: SCMTraceabilityData): Option[Integer] = {
    import com.xebialabs.deployit.repository.sql.base.SCMTraceabilityDBData._
    val keyHolder = new GeneratedKeyHolder()
    jdbcTemplate.update((conn: Connection) => {
      val ps = conn.prepareStatement(SCM_TRACEABILITY_DATA_INSERT, Array(ID.name))
      fillInTraceabilityParams(ps, scmData)
    }, keyHolder)
    Some(keyHolder.getKey.intValue())
  }

  override def update(traceabilityId: Integer, scmData: SCMTraceabilityData): Option[Integer] = {
    jdbcTemplate.update({ con: Connection =>
      val ps = con.prepareStatement(SCM_TRACEABILITY_DATA_UPDATE)
      fillInTraceabilityParams(ps, scmData)
      ps.setInt(8, traceabilityId)
      ps
    })
    Some(traceabilityId)
  }

  override def delete(id: Integer): Option[Integer] = {
    jdbcTemplate.update(SCM_TRACEABILITY_DATA_DELETE, id)
    None
  }
}
