package com.xebialabs.deployit.repository.sql.lock

import com.xebialabs.deployit.core.sql.{ColumnName, Queries, SchemaInfo, TableName}
import com.xebialabs.deployit.engine.tasker.TaskId
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.sql.lock.model.CiLock
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.dao.DuplicateKeyException
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional

import java.sql.ResultSet
import java.util.{Date, Set => JSet}
import scala.jdk.CollectionConverters._

trait CiLockRepository {
  def listAllLocks(): List[String]

  def listLocksByTaskId(taskId: TaskId): List[String]

  def clearLocksByTaskId(taskId: TaskId): Unit

  def clearAllLocks(): Unit

  def lock(ci: ConfigurationItem, taskId: TaskId): Boolean

  def lock(cis: JSet[ConfigurationItem], taskId: TaskId): Boolean

  def unlock(cis: JSet[ConfigurationItem], taskId: TaskId): Unit
}

@Component
@Transactional("mainTransactionManager")
class CiLockRepositoryImpl(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate)
                          (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends CiLockRepository with CiLockQueries with Logging {

  override def listLocksByTaskId(taskId: TaskId): List[String] =
    jdbcTemplate.query(LIST_LOCKS_BY_TASK_ID, (rs: ResultSet, _: Int) => rs.getString(1), taskId).asScala.toList

  override def listAllLocks(): List[String] =
    jdbcTemplate.query(LIST_ALL_LOCKS, (rs: ResultSet, _: Int) => rs.getString(1)).asScala.toList

  override def clearLocksByTaskId(taskId: String): Unit =
    jdbcTemplate.update(CLEAR_LOCKS_WITH_TASK_ID, taskId)

  override def clearAllLocks(): Unit =
    jdbcTemplate.update(CLEAR_ALL_LOCKS)

  def getLocks(ci: ConfigurationItem): List[CiLock] =
    jdbcTemplate.query(GET_LOCKS_BY_CI_ID, new RowMapper[CiLock] {
      override def mapRow(rs: ResultSet, rowNum: Int): CiLock =
        CiLock(rs.getString(1), rs.getString(2), rs.getInt(3))
    }, ci.getId).asScala.toList

  def lock(ci: ConfigurationItem, taskId: TaskId): Boolean = {

    val existingCiLocks: List[CiLock] = getLocks(ci)
    var seq = 1
    if (existingCiLocks.nonEmpty) {
      val result = canReentrant(ci, taskId, existingCiLocks)
      if (result.nonEmpty) {
        return result.get
      }
      seq = existingCiLocks.maxBy(_.seq).seq + 1
    }
    try {
      val res = jdbcTemplate.update(LOCK, ci.getId, taskId, seq)
      logger.info(s"Locking ${ci.getName} on ${new Date()}")
      if (res == 0) {
        logger.info(s"${ci.getName} already has a lock")
      }
      res > 0
    } catch {
      case _ : DuplicateKeyException =>
        logger.warn(s"${ci.getName} is locked by another task")
        false
      case e: Exception =>
        logger.warn(s"Failed to lock ${ci.getName} for task $taskId", e)
        false
    }
  }

  private def canReentrant(ci: ConfigurationItem, taskId: TaskId, existingCiLocks: List[CiLock]): Option[Boolean] = {
    if (existingCiLocks.exists(ciLock => taskId == ciLock.taskId)) {
      logger.info(s"Current task ${taskId} has already locked ${ci.getName}")
      Option(true)
    } else {
      if (ci.hasProperty("limitNumberOfConcurrentDeployments")) {
        val limit = ci.getProperty("limitNumberOfConcurrentDeployments").asInstanceOf[Int]
        if (existingCiLocks.size >= limit) {
          logger.warn(s"${ci.getName} is locked by another task : ${existingCiLocks.map(_.taskId).mkString(",")}")
          return Option(false)
        }
        Option.empty
      } else {
        logger.warn(s"${ci.getName} is locked by another task : ${existingCiLocks.map(_.taskId).mkString(",")}")
        Option(false)
      }
    }
  }

  override def lock(cis: JSet[ConfigurationItem], taskId: TaskId): Boolean =
    cis.asScala
      .forall(ci => lock(ci, taskId))

  override def unlock(cis: JSet[ConfigurationItem], taskId: TaskId): Unit =
    cis.asScala
      .filter(_.getId != null)
      .foreach(ci => {
        jdbcTemplate.update(UNLOCK, ci.getId, taskId)
      })
}

object CiLockSchema {
  val tableName: TableName = TableName("XLD_CI_LOCK")
  val ci_id: ColumnName = ColumnName("CI_ID")
  val task_id: ColumnName = ColumnName("task_id")
  val seq: ColumnName = ColumnName("seq")
}

trait CiLockQueries extends Queries {

  import CiLockSchema._

  val LIST_LOCKS_BY_TASK_ID = sqlb"select $ci_id from $tableName where $task_id = ?"
  val LIST_ALL_LOCKS = sqlb"select $ci_id from $tableName"

  val CLEAR_LOCKS_WITH_TASK_ID = sqlb"delete from $tableName where $task_id = ?"
  val CLEAR_ALL_LOCKS = sqlb"delete from $tableName"

  val LOCK = sqlb"insert into $tableName ($ci_id, $task_id, $seq) values (?, ?, ?)"
  val GET_LOCKS_BY_CI_ID = sqlb"select $ci_id, $task_id, $seq from $tableName where $ci_id = ?"

  val UNLOCK = sqlb"delete from $tableName where $ci_id = ? and $task_id = ?"
}

