package com.xebialabs.deployit.repository.sql.artifacts

import com.xebialabs.deployit.core.sql.sugar._
import com.xebialabs.deployit.core.sql.{JoinBuilder, SchemaInfo, SelectBuilder, SqlCondition => cond, SqlFunction => func}
import com.xebialabs.deployit.repository.sql.base._
import com.xebialabs.deployit.repository.sql.persisters.SourceArtifactSchema
import com.xebialabs.deployit.sql.base.schema.CIS

case class SourceArtifactQueryParams(parentIds: List[String] = Nil,
                                     supportedExtensions: List[String] = Nil,
                                     paging: Option[(Int, Int)] = None,
                                     order: List[(String, Boolean)] = Nil,
                                     count: Boolean = false,
                                     ciIds: List[String] = Nil)

object SourceArtifactQueryBuilder {
  private val CIS_ALIAS = "cis"
  private val SOURCE_ARTIFACTS_ALIAS = "sa"

  private val orderFieldsMapping = Map(
    "filename" -> SourceArtifactSchema.filename.tableAlias(SOURCE_ARTIFACTS_ALIAS),
    "path" -> CIS.path.tableAlias(CIS_ALIAS)
  )

  private def buildCiSelect(params: SourceArtifactQueryParams)(implicit schemaInfo: SchemaInfo): SelectBuilder = {
    val cisSelect = new SelectBuilder(CIS.tableName).as(CIS_ALIAS)
    val select = if (params.count) cisSelect.select(func.countAll) else cisSelect.select(CIS.path)

    val filterByParentId = (id: String) => {
      val fixedId = idToPath(if (id.endsWith("/")) id.dropRight(1) else id)
      cond.like(CIS.path, s"$fixedId/%")
    }

    select
      .withDisjunctionFilters[String](cond.equals(CIS.path, _), params.ciIds)
      .withDisjunctionFilters(filterByParentId, params.parentIds)
  }

  private def filterByFormat(format: String) =
    cond.like(func.lower(SourceArtifactSchema.filename), s"%.$format")

  private def buildSourceArtifactSelect(params: SourceArtifactQueryParams)(implicit schemaInfo: SchemaInfo) = {
    val sourceArtifactSelect = new SelectBuilder(SourceArtifactSchema.tableName).as(SOURCE_ARTIFACTS_ALIAS)
    val select = if (params.count) sourceArtifactSelect else sourceArtifactSelect.select(SourceArtifactSchema.filename)
    select
      .withDisjunctionFilters(filterByFormat, params.supportedExtensions)
  }

  def build(params: SourceArtifactQueryParams)(implicit schemaInfo: SchemaInfo): JoinBuilder = {
    val condition = cond.equals(CIS.ID.tableAlias(CIS_ALIAS), SourceArtifactSchema.ID.tableAlias(SOURCE_ARTIFACTS_ALIAS))
    buildCiSelect(params)
      .applyOrdering(orderFieldsMapping, params.order)
      .rightJoin(buildSourceArtifactSelect(params), condition)
      .applyPagination(params.paging)
  }
}
