package com.xebialabs.deployit.repository.sql.reader

import com.xebialabs.deployit.core.sql.spring.{MapRowMapper, Setter}
import com.xebialabs.deployit.core.sql.util.queryWithInClausePreserveOrder
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.repository.sql.base.{CiPKType, CiQueries, asCiPKType, pathToId}
import com.xebialabs.deployit.repository.sql.cache.{CiAppCacheContext, CiCacheDataServiceFacade}
import com.xebialabs.deployit.repository.sql.properties.CiPropertiesQueries
import com.xebialabs.deployit.repository.sql.reader.properties.CiPropertiesWithRefsProvider
import com.xebialabs.deployit.sql.base.schema.CIS
import grizzled.slf4j.Logging
import org.springframework.dao.EmptyResultDataAccessException
import org.springframework.jdbc.core.JdbcTemplate

import java.util
import scala.collection.mutable
import scala.jdk.CollectionConverters._

trait BaseCiReader extends CiQueries with CiPropertiesQueries with CiPropertiesWithRefsProvider with Logging {

  implicit val jdbcTemplate: JdbcTemplate
  implicit val ciCacheDataServiceFacade: CiCacheDataServiceFacade

  def readCiMapByPath(path: String): util.Map[String, Object] = {
    if (CiAppCacheContext.useCache) {
      logger.trace(s"using ci app cache for path($path)")
      ciCacheDataServiceFacade.getWithFallback(path, (p: String) => {
        queryCiByPath(p)
      })
    } else {
      queryCiByPath(path)
    }
  }

  def readCisMapsByPaths(group: Seq[String]): List[util.Map[String, Object]] = {
    if (CiAppCacheContext.useCache) {
      logger.trace(s"using ci app cache for paths($group)")
      ciCacheDataServiceFacade.getAllPathsWithFallback(group, (ciPaths: Seq[String]) => {
        queryCiByPaths(ciPaths)
      })
    } else {
      queryCiByPaths(group).toList
    }
  }

  def readCiMapByPk(pk: CiPKType): util.Map[String, Object] = {
    if (CiAppCacheContext.useCache) {
      logger.trace(s"using ci app cache for pk($pk)")
      ciCacheDataServiceFacade.getWithFallback(pk, (key: CiPKType) => {
        queryCiByPk(key)
      })
    } else {
      queryCiByPk(pk)
    }
  }

  def readCisMapsByPks(pks: Seq[CiPKType]): List[util.Map[String, Object]] = {
    if (CiAppCacheContext.useCache) {
      logger.trace(s"using ci app cache for pks($pks)")
      ciCacheDataServiceFacade.getAllPksWithFallback(pks, (keys: Seq[CiPKType]) => {
        queryWithInClausePreserveOrder[CiPKType, util.Map[String, Object]](keys, map => asCiPKType(map.get(CIS.ID.name))) { group =>
          queryCiByPks(group)
        }
      })
    } else {
      queryWithInClausePreserveOrder[CiPKType, util.Map[String, Object]](pks, map => asCiPKType(map.get(CIS.ID.name))) { group =>
        queryCiByPks(group)
      }
    }
  }
  def queryCiByPath(path: String): util.Map[String, Object] = {
    try {
      jdbcTemplate.queryForObject(SELECT_CI_BY_PATH, MapRowMapper, path)
    } catch {
      case _: EmptyResultDataAccessException =>
        throw new NotFoundException("Repository entity [%s] not found", pathToId(path))
    }
  }

  def queryCiByPaths(group: Seq[String]): mutable.Buffer[util.Map[String, Object]] = {
    jdbcTemplate
      .query(
        buildSelectCisByPathsQuery(group),
        Setter(group),
        MapRowMapper
      ).asScala
  }

  def queryCiByPk(pk: CiPKType): util.Map[String, Object] = {
    jdbcTemplate.queryForObject(SELECT_CI_BY_ID, MapRowMapper, pk)
  }

  def queryCiByPks(group: Seq[CiPKType]): mutable.Buffer[util.Map[String, Object]] = {
    jdbcTemplate
      .query[util.Map[String, Object]](
        buildSelectCisByIdsQuery(group),
        Setter(group),
        MapRowMapper
      ).asScala
  }
}
