package com.xebialabs.deployit.repository.sql.reader.properties

import com.xebialabs.deployit.core.sql.SchemaInfo
import com.xebialabs.deployit.core.sql.spring.{MapRowMapper, Setter}
import com.xebialabs.deployit.core.sql.util.queryWithInClause
import com.xebialabs.deployit.repository.sql.base.{CiPKType, CiQueries, asCiPKType}
import com.xebialabs.deployit.repository.sql.cache.CiCacheDataServicesHolder
import com.xebialabs.deployit.repository.sql.properties.CiPropertiesQueries
import com.xebialabs.deployit.sql.base.schema.{CIS, CI_PROPERTIES}
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.JdbcTemplate

import java.util
import java.util.stream.Collectors
import java.util.{ArrayList => JArrayList, List => JList, Map => JMap, Set => JSet, TreeMap => TMap}
import scala.jdk.CollectionConverters._

trait CiPropertiesWithRefsProvider extends CiPropertiesQueries with CiQueries with Logging {

  implicit val jdbcTemplate: JdbcTemplate
  implicit val schemaInfo: SchemaInfo

  def getCiPropertiesWithRefs(pk: CiPKType): JList[JMap[String, AnyRef]] = {
    val ciProperties =
      CiCacheDataServicesHolder
        .getCiCacheDataServiceFacade
        .getPropertiesByPkWithFallback(pk, (p: CiPKType) => {
          jdbcTemplate.query(SELECT_PROPERTIES, MapRowMapper, p)
        })
    val cis = getCisByPks(
      collectCiRefs(Map(pk -> ciProperties).asJava).asScala.toList
    ).groupBy(map => asCiPKType(map.get(CIS.ID.name)))
    bundleRefValues(ciProperties, cis)
  }

  def getCiPropertiesWithRefs(ciPks: Seq[CiPKType]): JMap[CiPKType, JList[JMap[String, AnyRef]]] = {
    val result = new TMap[CiPKType, JList[JMap[String, AnyRef]]]()
    val map =
      CiCacheDataServicesHolder
        .getCiCacheDataServiceFacade
        .getAllPropertiesWithFallback(ciPks, (pks: Seq[CiPKType]) => {
          queryWithInClause(pks) { group =>
            jdbcTemplate.query(buildSelectCisPropertiesByIdsQuery(group), Setter(group), MapRowMapper).asScala
          }.groupBy(map => asCiPKType(map.get(CI_PROPERTIES.ci_id.name)))
        })
    val cis = getCisByPks(
      collectCiRefs(map).asScala.toList
    ).groupBy(map => asCiPKType(map.get(CIS.ID.name)))
    map.entrySet().forEach(e => result.put(e.getKey, bundleRefValues(e.getValue, cis)))
    result
  }

  private def bundleRefValues(ciProperties: JList[JMap[String, AnyRef]],
                                       cis: Map[CiPKType, List[JMap[String, AnyRef]]]):
  JList[JMap[String, AnyRef]] = {
    val result = new JArrayList[JMap[String, AnyRef]]()
    ciProperties.forEach(map => {
      if (map != null && map.containsKey(CI_PROPERTIES.ci_ref_value.name)) {
        val ciRefValue = map.get(CI_PROPERTIES.ci_ref_value.name).asInstanceOf[CiPKType]
        if (ciRefValue != null && cis.contains(ciRefValue)) {
          cis(ciRefValue).head.entrySet.forEach(m => {
            map.put(s"a_${m.getKey}", m.getValue)
          })
        } else {
          CIS.allFields.foreach(column => {
            map.put(s"a_${column.name}", null)
          })
        }
      }
      result.add(map)
    })
    result
  }

  private def collectCiRefs(properties: JMap[CiPKType, JList[JMap[String, AnyRef]]]): JSet[CiPKType] = {
    properties
      .values().stream().flatMap(_.stream())
      .filter(
        map => map != null &&
          map.containsKey(CI_PROPERTIES.ci_ref_value.name) &&
          map.get(CI_PROPERTIES.ci_ref_value.name) != null
      )
      .map(map => map.get(CI_PROPERTIES.ci_ref_value.name).asInstanceOf[CiPKType])
      .collect(Collectors.toSet[CiPKType])
  }

  private def getCisByPks(pks: Seq[CiPKType]): List[JMap[String, Object]] = {
    CiCacheDataServicesHolder
      .getCiCacheDataServiceFacade
      .getAllPksWithFallback(pks, (keys: Seq[CiPKType]) => {
        queryWithInClause[CiPKType, util.Map[String, Object]](keys) { group =>
          jdbcTemplate.query[util.Map[String, Object]](
            buildSelectCisByIdsQuery(group),
            Setter(group),
            MapRowMapper
          ).asScala
        }
      })
  }

}
