package com.xebialabs.deployit.repository.sql.reader.properties

import com.xebialabs.deployit.core.sql.SchemaInfo
import com.xebialabs.deployit.core.sql.spring.{MapRowMapper, Setter}
import com.xebialabs.deployit.core.sql.util.queryWithInClause
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.sql.base._
import com.xebialabs.deployit.repository.sql.cache.{CiAppCacheContext, CiCacheDataServicesHolder}
import com.xebialabs.deployit.repository.sql.properties.{CiPropertiesQueries, PropertyTableRepository}
import com.xebialabs.deployit.repository.sql.specific.TypeSpecificPersisterFactory
import com.xebialabs.deployit.repository.sql.specific.configurable.PropertyTableSelectDescriptor
import com.xebialabs.deployit.sql.base.schema.CI_PROPERTIES
import org.springframework.jdbc.core.JdbcTemplate

import java.util.{ArrayList => JArrayList, List => JList, Map => JMap}
import scala.jdk.CollectionConverters._

/**
 * Data provider that batches retrieval of multiple CI data into one query.
 * It will load (on demand), group and cache results.
 * This approach allows us to significantly reduce an amount of requests to DB when reading multiple CIs within one
 * transaction.
 *
 * @param ciIds        - list of configuration item ids
 * @param jdbcTemplate - `org.springframework.jdbc.core.JdbcTemplate` instance
 * @param schemaInfo   - `com.xebialabs.deployit.core.sql.SchemaInfo` instance
 */
class CiGroupCiDataProvider(ciIds: Seq[(CiPKType, Type)])(implicit val jdbcTemplate: JdbcTemplate,
                                                          val schemaInfo: SchemaInfo,
                                                          typeSpecificPersisterFactories: JList[TypeSpecificPersisterFactory])
  extends CiDataProvider with CiPropertiesQueries with LookupValuesQueries with PropertyTableRepository
   with CiPropertiesWithRefsProvider {
  private def queryAndGroup(buildQuery: Seq[CiPKType] => String): Map[CiPKType, List[JMap[String, AnyRef]]] =
    queryWithInClause(ciIds.map { case (id, _) => id }) { group =>
      jdbcTemplate.query(buildQuery(group), Setter(group), MapRowMapper).asScala
    }.groupBy(map => asCiPKType(map.get(CI_PROPERTIES.ci_id.name)))

  private lazy val ciProperties: Either[JMap[CiPKType, JList[JMap[String, AnyRef]]], Map[CiPKType, List[JMap[String, AnyRef]]]] = {
    if (CiAppCacheContext.useCache && CiCacheDataServicesHolder.getCachesProperties.appCaches.enabled) {
      Left(getCiPropertiesWithRefs(ciIds.map { case (id, _) => id }))
    } else {
      Right(queryAndGroup(buildSelectPropertiesWithRefsByCiIds))
    }
  }

  private lazy val lookupValues = queryAndGroup(buildSelectLookupValuesByCiIdsQuery)

  private def groupSelectDescriptorsWithCiIds: Map[PropertyTableSelectDescriptor, Seq[CiPKType]] = {
    val seq = for {
      (pk, ciType) <- ciIds
      reader <- typeSpecificPersisterFactories.asScala.flatMap(_.createReader(ciType, pk))
      tableDescriptor <- reader.tables.values
      selectDescriptor = tableDescriptor.selectDescriptor
    } yield (selectDescriptor, pk)
    seq
      .groupBy { case (descriptor, _) => descriptor }
      .view
      .mapValues(_.map { case (_, pk) => pk })
      .toMap
  }

  private lazy val propertyTableValues: Map[PropertyTableSelectDescriptor, Map[CiPKType, JList[JMap[String, AnyRef]]]] =
    groupSelectDescriptorsWithCiIds.map { case (descriptor, pks) =>
      descriptor -> readPropertyTableForCis(pks, descriptor)
        .groupBy(map => asCiPKType(map.get(s"p_${descriptor.ciIdColumn.name}")))
        .view
        .mapValues(_.asJava)
        .toMap
    }

  override def getProperties(id: CiPKType): JList[JMap[String, AnyRef]] = {
    ciProperties match {
      case Left(v) => v.getOrDefault(id, new JArrayList[JMap[String, AnyRef]]())
      case Right(v) => v.getOrElse(id, Nil).asJava
    }
  }

  override def getLookupValues(id: CiPKType): JList[JMap[String, AnyRef]] = lookupValues.getOrElse(id, Nil).asJava

  override def getPropertyTableValues(id: CiPKType,
                                      selectDescriptor: PropertyTableSelectDescriptor): JList[JMap[String, AnyRef]] =
    propertyTableValues.get(selectDescriptor).flatMap(_.get(id)).getOrElse(new JArrayList[JMap[String, AnyRef]]())
}
