package com.xebialabs.deployit.repository.sql.commands

import com.xebialabs.deployit.booter.local.CiRoots
import com.xebialabs.deployit.checks.Checks
import com.xebialabs.deployit.core.util.CiSugar._
import com.xebialabs.deployit.core.util.CiValidationUtils._
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind.{LIST_OF_CI, SET_OF_CI}
import com.xebialabs.deployit.plugin.api.reflect._
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot
import com.xebialabs.deployit.repository.core.Directory
import com.xebialabs.deployit.repository.internal.Root
import com.xebialabs.deployit.repository.sql.base._
import com.xebialabs.deployit.repository.sql.commands.CiPathValidation.CiValidationGroups

import scala.jdk.CollectionConverters._

object CiPathValidation {

  case class CiValidationGroups(directories: Vector[(String, Type)] = Vector(),
                                nestedCis: Vector[(String, Type)] = Vector(),
                                rest: Vector[(String, Type)] = Vector())

}

trait CiPathValidation extends CiQueries with CiExists {
  private def isAssignable(descriptor: Descriptor)(pd: PropertyDescriptor): Boolean =
    descriptor.isAssignableTo(pd.getReferencedType)

  private def checkIsAssignable(filterPredicate: PropertyDescriptor => Boolean,
                                sourceDescriptor: Descriptor,
                                targetDescriptor: Descriptor): Boolean =
    sourceDescriptor
      .getPropertyDescriptors
      .asScala
      .filter(filterPredicate)
      .exists(isAssignable(targetDescriptor))

  private def checkParentIsAssignable(childDescriptor: Descriptor, parentDescriptor: Descriptor): Boolean =
    checkIsAssignable(pd => pd.getKind == PropertyKind.CI && pd.isAsContainment, childDescriptor, parentDescriptor)

  private def checkChildIsAssignable(childDescriptor: Descriptor, parentDescriptor: Descriptor): Boolean =
    checkIsAssignable(pd => (pd.getKind == SET_OF_CI || pd.getKind == LIST_OF_CI) && pd.isAsContainment, parentDescriptor, childDescriptor)

  private def checkCommonCiAttributes(id: String, ciType: Type): Unit = {
    Checks.checkArgument(DescriptorRegistry.exists(ciType), "Unknown configuration item type %s", ciType)
    id.pathElements.foreach(validateName)
    validateId(id)
  }

  /**
   * Performs validation of directory CIs
   * It mainly makes sure that directory id is correct
   *
   * @param cis - vector of directory id to type mappings
   */
  private def validateDirectories(cis: Vector[(String, Type)]): Unit = {
    cis.foreach { case (id, _) =>
      val pathElements = id.pathElements
      Checks.checkArgument(pathElements.length >= 2, "A core.Directory must be stored under a Root node, not under %s.", id)
      pathElements.headOption.foreach { root =>
        Checks.checkArgument(CiRoots.all().asScala.toSet.contains(root), "A core.Directory must be stored under a valid Root node, not under %s.", id)
      }
    }
  }

  /**
   * This function validates generic CI types (that are not directories, root nodes or nested/embedded CIs)
   *
   * It performs 2 validations:
   * 1. Makes sure that CI is stored where it is designed to be stored (proper root node)
   * 2. Makes sure that configuration item's parent exists
   *
   * @param cis - vector of ci ids to ci type relations
   */
  private def validateRest(cis: Vector[(String, Type)])(implicit context: ChangeSetContext): Unit = {
    val loadedCachedPaths = context.ciCache.values.to(LazyList).map(_.getId).map(idToPath).toSet

    val pathsToLoad = cis
      .to(LazyList)
      .flatMap { case (id, ciType) =>
        val desc = ciType.descriptor
        id.pathElements.headOption.foreach { root =>
          Checks.checkArgument(root == desc.getRootName, "Configuration item of type %s cannot be stored at %s. It should be stored under %s", ciType, id, desc.getRootName)
        }
        id.allPaths
      }
      .filterNot(loadedCachedPaths.contains)
      .distinct

    val pathsThatExist = selectExistingPaths(pathsToLoad) ++ loadedCachedPaths

    cis.foreach { case (id, ciType) =>
      id.allPaths.foreach { path =>
        Checks.checkArgument(path == idToPath(id) || pathsThatExist.contains(path), "Configuration item of type %s cannot be stored at %s. The parent does not exist", ciType, id)
      }
    }
  }

  /**
   * Builds index of CI paths to parent type first by looking at ChangeSetContext and then, by loading the information
   * from database for CIs, that was not found in ChangeSetContext
   *
   * @param paths   - set of CI paths
   * @param context - changeset context instance
   * @return map of CI path to ci Type relations
   */
  private def createCiTypesMap(paths: Set[String])(implicit context: ChangeSetContext): Map[String, Type] = {
    val parentTypesFromContext = paths.flatMap(path => context.ciCache.get(pathToId(path)).map(ci => (path, ci.getType))).toMap
    val pathsToLoad = paths.filterNot(parentTypesFromContext.contains).toSeq
    val parentTypesFromDB = if (pathsToLoad.nonEmpty) selectExistingCiTypes(pathsToLoad) else Map()
    parentTypesFromContext ++ parentTypesFromDB
  }

  /**
   * Validates nested (embedded) configuration items
   *
   * This function will:
   * 1. Make sure that embedded CIs are stored in a right place (depth must be at least 3)
   * 2. Check for parent CI existence
   * 3. Check if nested CI can be stored under the parent that is specified in CI id
   *
   * @param cis     - vector of ci id to ci type relations
   * @param context - instance of `com.xebialabs.deployit.repository.sql.commands.ChangeSetContext`
   */
  private def validateNestedCis(cis: Vector[(String, Type)])(implicit context: ChangeSetContext): Unit = {
    val parentPaths = cis
      .flatMap { case (id, ciType) =>
        Checks.checkArgument(id.pathElements.length >= 3, "Configuration item of type %s cannot be stored at %s. It should be stored under a valid parent node", ciType, id)
        parentPath(id)
      }
      .toSet

    val parentTypes = createCiTypesMap(parentPaths)
    cis.foreach { case (id, ciType) =>
      parentPath(id).foreach { parentPath =>
        val parentType = parentTypes.getOrElse(
          parentPath,
          throw new Checks.IncorrectArgumentException("Configuration item of type %s cannot be stored at %s. The parent does not exist", ciType, id)
        )
        val parentDescriptor = parentType.descriptor
        val desc = ciType.descriptor
        Checks.checkArgument(
          checkParentIsAssignable(desc, parentDescriptor) || checkChildIsAssignable(desc, parentDescriptor),
          "Configuration item of type %s cannot be stored at %s. The parent cannot contain configuration items of this type", ciType, id
        )
      }
    }
  }

  /**
   * Validates configuration items before they are being persisted.
   * There are 3 groups of configuration items that are being validated by this function:
   * 1. Directories
   * 2. Nested CIs
   * 3. All the rest
   *
   * There is also an exceptional case (root) which is not actually validated but is simply filtered out
   *
   * @param cis     - iterable of ci ids to ci types tuples
   * @param context - instance of `com.xebialabs.deployit.repository.sql.commands.ChangeSetContext`
   */
  protected def validateCisStoredInCorrectPath(cis: Iterable[(String, Type)])(implicit context: ChangeSetContext): Unit = {
    val validationGroups = cis.foldLeft(CiValidationGroups()) { case (acc, current@(id, ciType)) =>
      checkCommonCiAttributes(id, ciType)
      ciType.descriptor match {
        case desc if desc.isAssignableTo(classOf[Root]) => acc
        case desc if desc.isAssignableTo(classOf[Directory]) => acc.copy(directories = current +: acc.directories)
        case desc if desc.getRoot == ConfigurationItemRoot.NESTED => acc.copy(nestedCis = current +: acc.nestedCis)
        case _ => acc.copy(rest = current +: acc.rest)
      }
    }
    validateDirectories(validationGroups.directories)
    validateNestedCis(validationGroups.nestedCis)
    validateRest(validationGroups.rest)
  }
}
