package com.xebialabs.deployit.repository.sql

import java.sql.ResultSet
import java.util
import java.util.Calendar

import com.fasterxml.jackson.databind.ObjectMapper
import com.xebialabs.deployit.core.sql._
import com.xebialabs.deployit.core.sql.batch.{BatchCommand, BatchCommandWithArgs}
import com.xebialabs.deployit.exception.UnknownRevisionException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.sql.CiHistoryRepository._
import com.xebialabs.deployit.repository.sql.base.{CiPKType, asCiPKType, idToPath}
import com.xebialabs.deployit.sql.base.schema.CIS
import com.xebialabs.deployit.repository.{ConfigurationItemRevision, HistoryService}
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Repository
import org.springframework.transaction.annotation.Transactional

trait CiHistoryRepository {

  def getVersionRevisions(id: String): util.List[ConfigurationItemRevision]

  def readRevision[T <: ConfigurationItem](id: String, revisionName: String): T

  def batchInsert(pk: CiPKType, ci: ConfigurationItem, now: DateTime, userName: String): Option[BatchCommandWithArgs]

  def delete(pk: CiPKType): Unit
}

@Repository
@Transactional("mainTransactionManager")
class CiHistoryRepositoryImpl (@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                               @Autowired @Qualifier("historyMapper") val mapper: ObjectMapper)
                              (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends CiHistoryRepository with HistoryQueries {

  override def getVersionRevisions(id: String): util.List[ConfigurationItemRevision] = {
    createRevisions(fetchRevisionData(id))
  }

  private def fetchRevisionData(id: String): util.List[(Int, CiPKType, Calendar, String)] = {
    jdbcTemplate.query(
      SELECT_REVISIONS,
      (rs: ResultSet, row: Int) => {
        val pk = asCiPKType(rs.getInt(1))
        val timestamp = toCalendar(rs.getTimestamp(2, UTC_CALENDAR))
        val username = rs.getString(3)
        (row, pk, timestamp, username)
      },
      idToPath(id)
    )
  }

  private def createRevisions(items: util.List[(Int, CiPKType, Calendar, String)]) = {
    val result = new util.ArrayList[ConfigurationItemRevision](items.size())
    val itr = items.iterator()
    while (itr.hasNext) {
      val (row, _, timestamp, username) = itr.next()
      result.add(
        if (itr.hasNext) {
          new ConfigurationItemRevision(indexToRevisionName(row), timestamp, username)
        } else {
          new ConfigurationItemRevision(HistoryService.CURRENT_REVISION, timestamp, username)
        }
      )
    }
    result
  }

  override def readRevision[T <: ConfigurationItem](id: String, revisionName: String): T = {
    def throwUnknownRevision = {
      throw new UnknownRevisionException("Cannot find revision [%s] for configuration item [%s]", revisionName, id)
    }

    val idx: Int = revisionName match {
      case nameRegex(res) => res.toInt
      case _ => throwUnknownRevision
    }

    val revisions = fetchRevisionData(id)
    if (revisions.size() < idx) throwUnknownRevision
    val pk = revisions.get(idx)._2
    val result = jdbcTemplate.queryForList(SELECT_CI, classOf[String], pk, idToPath(id))
    val ci_json: String = result.get(0)
    mapper.readValue(ci_json, classOf[ConfigurationItem]).asInstanceOf[T]
  }

  override def batchInsert(pk: CiPKType, ci: ConfigurationItem, now: DateTime, userName: String): Option[BatchCommandWithArgs] = {
    if (shouldVersion(ci.getId, ci.getType)) {
      Some(BatchCommand(
        INSERT,
        pk,
        toCalendar(now),
        Option(userName).getOrElse("<system>"),
        mapper.writerFor(classOf[ConfigurationItem]).writeValueAsString(ci)
      ))
    } else
      None
  }

  override def delete(pk: CiPKType): Unit =
    jdbcTemplate.update(DELETE, pk)
}

object CiHistoryRepository {

  def indexToRevisionName(idx: Int) = s"1.$idx"

  final val nameRegex = """1\.(\d*)""".r

  def shouldVersion(id: String, ciType: Type): Boolean =
    !id.startsWith("Applications/") && ciType.getDescriptor.isVersioned

}

object HistorySchema {
  val tableName = TableName("XLD_CI_HISTORY")

  val ID = ColumnName("ID")
  val ci_id = ColumnName("ci_id")
  val changed_at = ColumnName("changed_at")
  val changed_by = ColumnName("changed_by")
  val ci = ColumnName("ci")
}

trait HistoryQueries extends Queries {

  import HistorySchema._

  lazy val INSERT =
    sqlb"insert into $tableName ($ci_id, $changed_at, $changed_by, $ci) values (?,?,?,?)"

  lazy val DELETE =
    sqlb"delete from $tableName where $ci_id = ?"

  lazy val SELECT_CI =
    sqlb"""select $ci from $tableName where $ID = ? and $ci_id =
          |(select ${CIS.ID} from ${CIS.tableName} where ${CIS.path} = ?)"""

  lazy val SELECT_REVISIONS =
    sqlb"""select $ID, $changed_at, $changed_by from $tableName where $ci_id =
          |(select ${CIS.ID} from ${CIS.tableName} where ${CIS.path} = ?)
          |order by $changed_at, $ID asc"""

}
