package com.xebialabs.deployit.repository.sql.artifacts

import ai.digital.configuration.central.deploy.db.Artifacts
import com.xebialabs.deployit.checksum.ChecksumAlgorithmProvider
import com.xebialabs.deployit.util.JavaCryptoUtils
import grizzled.slf4j.Logging
import org.apache.commons.io.FileUtils
import org.springframework.transaction.support.{TransactionSynchronization, TransactionSynchronizationManager}

import java.io.{File, InputStream}
import java.security.DigestInputStream


trait TmpFileArtifactHandler extends Logging{

  val root: File

  val defaultRoot: File = new File(Artifacts.DEFAULT_ROOT)

  if (defaultRoot != null) defaultRoot.mkdirs()

  private def tmpFileDir = if (defaultRoot == null || root.getAbsolutePath.equals(defaultRoot.getAbsolutePath)) root else defaultRoot

  def storeToTempWithChecksumCalc(inputStream: InputStream)(implicit checksumAlgorithmProvider: ChecksumAlgorithmProvider): (File, String) = {
    val digest = checksumAlgorithmProvider.getMessageDigest
    val digestInputStream = new DigestInputStream(inputStream, digest)
    val tempFile = storeToTemp(digestInputStream)
    val checksum = JavaCryptoUtils.digest(digest)
    debug(s"checksum $checksum for tmp file ${tempFile.getAbsolutePath}")
    (tempFile, checksum)
  }

  def storeToTemp(inputStream: InputStream): File = {
    val tempFile = File.createTempFile("artifact", "tmp", tmpFileDir)
    debug(s"the tmp file created in storeToTemp is ${tempFile.getAbsolutePath}")
    FileUtils.copyInputStreamToFile(inputStream, tempFile)
    tempFile
  }

  def storeToTempWithEval[T](inputStream: InputStream)(evalBlock: (InputStream, String) => T)
                         (implicit checksumAlgorithmProvider: ChecksumAlgorithmProvider): T = {
    val (tempFile, checksum) = storeToTempWithChecksumCalc(inputStream)
    val tempInputStream = readFromTemp(tempFile)
    TransactionSynchronizationManager.registerSynchronization(new TransactionSynchronization() {
      override def afterCompletion(status: Int): Unit = {
        tempInputStream.close()
        deleteFileFromTemp(tempFile)
      }
    })
    evalBlock(tempInputStream, checksum)
  }

  def readFromTemp(tempFile: File): InputStream = FileUtils.openInputStream(tempFile)

  def deleteFileFromTemp(tempFile: File): Boolean =
    if (tempFile.exists()) {
      tempFile.delete()
    } else {
      false
    }
}
