package com.xebialabs.deployit.repository.sql.base

import com.xebialabs.deployit.core.sql.batch.{BatchCommand, BatchCommandWithArgs}
import com.xebialabs.deployit.core.sql.spring.{MapRowMapper, Setter}
import com.xebialabs.deployit.core.sql.util.queryWithInClause
import com.xebialabs.deployit.core.sql.{Queries, SchemaInfo, SelectBuilder, SqlCondition => cond}
import com.xebialabs.deployit.sql.base.schema.CIS
import org.springframework.jdbc.core.JdbcTemplate

import java.sql.ResultSet
import scala.jdk.CollectionConverters._

trait SecuredCi extends SecuredCiQueries {
  val jdbcTemplate: JdbcTemplate

  def getSecuredCi(pk: CiPKType): Option[CiPKType] = jdbcTemplate.query(SELECT_SECURED_CI, (rs: ResultSet, _: Int) => asCiPKType(rs.getInt(1)), pk).asScala.headOption

  class CisSearchPathsQueryBuilder(private val pks: Iterable[CiPKType])
                                  (implicit schemaInfo: SchemaInfo) {

    val selectBuilder: SelectBuilder = new SelectBuilder(CIS.tableName)
      .select(CIS.ID)
      .select(CIS.secured_ci)
      .where(cond.in(CIS.ID, pks))
  }

  def getSecuredCis(pks: Iterable[CiPKType]): List[(CiPKType, CiPKType)] =
    queryWithInClause(pks.toSet.toSeq) { pkGroup =>
      val selectBuilder = new CisSearchPathsQueryBuilder(pkGroup).selectBuilder
      jdbcTemplate
        .query(selectBuilder.query, Setter(selectBuilder.parameters), MapRowMapper)
        .asScala
        .map { m =>
          val securedCi = Option(m.get(CIS.secured_ci.name))
          // getOrElse with 0 - it is expected 0 in case of null, as ResultSet.getInt is working like that
          (asCiPKType(m.get(CIS.ID.name)), asCiPKType(securedCi.getOrElse(0)))
        }
    }

  def setSecuredCi(pk: CiPKType, securedCi: CiPKType): Unit = jdbcTemplate.update(UPDATE_SECURED_CI, securedCi, pk)

  def batchSetSecuredCi(pk: CiPKType, securedCi: CiPKType): BatchCommandWithArgs = BatchCommand(UPDATE_SECURED_CI, securedCi, pk)

}

trait SecuredCiQueries extends Queries {

  lazy val SELECT_SECURED_CI: String = {
    import CIS._
    sqlb"select $secured_ci from $tableName where $ID = ?"
  }

  lazy val UPDATE_SECURED_CI: String = {
    import CIS._
    sqlb"update $tableName set $secured_ci = ? where $ID = ?"
  }

}
