package com.xebialabs.deployit.repository.sql.reader

import com.xebialabs.deployit.core.util.CiSugar._
import com.xebialabs.deployit.core.util.TypeConversions._
import com.xebialabs.deployit.core.sql.SchemaInfo
import com.xebialabs.deployit.core.sql.spring.{MapRowMapper, Setter}
import com.xebialabs.deployit.core.sql.util._
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.io.SourceArtifactFile
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind.{LIST_OF_STRING, MAP_STRING_STRING, SET_OF_STRING}
import com.xebialabs.deployit.plugin.api.reflect.{Descriptor, PropertyDescriptor, PropertyKind, Type}
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact
import com.xebialabs.deployit.plugin.api.udm.lookup.LookupValueKey
import com.xebialabs.deployit.plugin.api.udm.{CiAttributes, ConfigurationItem, ExternalProperty}
import com.xebialabs.deployit.repository.WorkDir
import com.xebialabs.deployit.repository.sql.artifacts.ArtifactRepository
import com.xebialabs.deployit.repository.sql.base.{CiQueries, _}
import com.xebialabs.deployit.repository.sql.properties.{CiPropertiesQueries, DeleteCiPropertiesQueries, EncryptedPropertyReader}
import com.xebialabs.deployit.repository.sql.reader.properties.{CiDataProvider, CiGroupCiDataProvider, OnDemandCiDataProvider}
import com.xebialabs.deployit.repository.sql.specific.configurable.ConfigurableTypeSpecificPersisterFactory.CiFromMapReader
import com.xebialabs.deployit.repository.sql.specific.{TypeSpecificPersisterFactory, TypeSpecificReader}
import com.xebialabs.deployit.sql.base.schema._
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlplatform.artifact.resolution.ArtifactResolverRegistry
import com.xebialabs.xlplatform.utils.ResourceManagement.using
import grizzled.slf4j.Logging
import org.apache.commons.io.FilenameUtils
import org.joda.time.DateTime
import org.springframework.dao.EmptyResultDataAccessException
import org.springframework.jdbc.core.JdbcTemplate

import java.util
import java.util.{List => JList, Map => JMap}
import scala.collection.mutable
import scala.jdk.CollectionConverters._

class CiReader(workDir: WorkDir,
               cache: mutable.HashMap[CiPKType, ConfigurationItem],
               implicit val jdbcTemplate: JdbcTemplate,
               passwordEncrypter: PasswordEncrypter,
               artifactRepository: ArtifactRepository,
               createTypeSpecificReaders: (Type, CiPKType) => List[TypeSpecificReader])
              (implicit val schemaInfo: SchemaInfo, val typeSpecificPersisterFactories: JList[TypeSpecificPersisterFactory])
  extends CiQueries with CiPropertiesQueries with DeleteCiPropertiesQueries with LookupValuesQueries with Logging with SCMTraceabilityDataQueries {

  private def extractCiFromMap(map: util.Map[String, Object],
                               depth: Int,
                               useCache: Boolean = true,
                               ciDataProvider: CiDataProvider): ConfigurationItem = {
    val pk = asCiPKType(map.get(CIS.ID.name))
    if (useCache) {
      readUsingCache(pk, depth)(map, ciDataProvider)
    } else {
      val item = readBaseCiFromMap(map)
      loadItemFromDb(item, pk, map, depth - 1, ciDataProvider)
    }
  }

  private def asCiIdsAndType(cis: Seq[JMap[String, AnyRef]]): Seq[(CiPKType, Type)] =
    cis.map(map => asCiPKType(map.get(CIS.ID.name)) -> map.get(CIS.ci_type.name).toString.ciType)

  def readByPaths(paths: Seq[String],
                  depth: Int,
                  useCache: Boolean = true,
                  skipNotExistingCis: Boolean = true): List[ConfigurationItem] = {
    val loaded = queryWithInClausePreserveOrder[String, ConfigurationItem](paths, ci => idToPath(ci.getId)) { group =>
      val cis = jdbcTemplate
        .query(buildSelectCisByPathsQuery(group), Setter(group), MapRowMapper)
        .asScala
      val ciDataProvider = new CiGroupCiDataProvider(asCiIdsAndType(cis.toSeq))
      cis.map(extractCiFromMap(_, depth, useCache, ciDataProvider))
    }
    if (!skipNotExistingCis) {
      val loadedPaths = loaded.map(_.getId).toSet
      paths
        .map(pathToId)
        .find(!loadedPaths.contains(_))
        .foreach(x => throw new NotFoundException("Repository entity [%s] not found", x))
    }
    loaded
  }

  def readByPath(path: String, depth: Int, useCache: Boolean = true): ConfigurationItem = try {
    val map = jdbcTemplate.queryForObject(SELECT_CI_BY_PATH, MapRowMapper, path)
    extractCiFromMap(map, depth, useCache, new OnDemandCiDataProvider)
  } catch {
    case _: EmptyResultDataAccessException =>
      throw new NotFoundException("Repository entity [%s] not found", pathToId(path))
  }

  def readByPk(pk: CiPKType, depth: Int): ConfigurationItem =
    readUsingCache(pk, depth)(jdbcTemplate.queryForObject(SELECT_CI_BY_ID, MapRowMapper, pk), new OnDemandCiDataProvider)

  def readByPks(pks: Seq[CiPKType], depth: Int): List[ConfigurationItem] = {
    val ciMaps = queryWithInClausePreserveOrder[CiPKType, util.Map[String, Object]](pks, map => asCiPKType(map.get(CIS.ID.name))) { group =>
      jdbcTemplate.query[util.Map[String, Object]](buildSelectCisByIdsQuery(group), Setter(group), MapRowMapper).asScala
    }
    val ciDataProvider = new CiGroupCiDataProvider(asCiIdsAndType(ciMaps))
    ciMaps.map(extractCiFromMap(_, depth, useCache = true, ciDataProvider))
  }

  private def readFromMap(map: util.Map[String, Object], depth: Int, ciDataProvider: CiDataProvider): ConfigurationItem = {
    val pk = asCiPKType(map.get(CIS.ID.name))
    readUsingCache(pk, depth)(map, ciDataProvider)
  }

  def readUsingCache(pk: CiPKType, depth: Int)(mapProvider: => util.Map[String, Object], ciDataProvider: CiDataProvider = new OnDemandCiDataProvider): ConfigurationItem =
    if (depth > 0) {
      cache.getOrElse(pk, {
        val map = mapProvider
        val item = readBaseCiFromMap(map)
        cache.put(pk, item)
        loadItemFromDb(item, pk, map, depth - 1, ciDataProvider)
      })
    } else {
      cache.getOrElse(pk, readBaseCiFromMap(mapProvider))
    }

  private def loadItemFromDb(item: ConfigurationItem,
                             pk: CiPKType,
                             map: util.Map[String, Object],
                             depth: Int,
                             ciDataProvider: CiDataProvider): ConfigurationItem = {
    asBaseConfigurationItem(item) { bci =>
      bci.set$token(map.get(CIS.token.name).asInstanceOf[String])
      bci.set$ciAttributes(newCiAttributes(map))
      bci.set$externalProperties(readExternalProperties(ciDataProvider.getLookupValues(pk)))
      bci.set$internalId(asCiPKType(map.get(CIS.ID.name)))
      bci.set$referenceId(map.get(CIS.reference_id.name).asInstanceOf[String])
      if (map.containsKey(CIS.secured_ci.name) && map.get(CIS.secured_ci.name) != null) {
        bci.set$securedCi(asCiPKType(map.get(CIS.secured_ci.name)))
      }
      bci.set$directoryReference(map.get(CIS.directory_ref.name).asInstanceOf[String])
      bci.set$securedDirectoryReference(map.get(CIS.secured_directory_ref.name).asInstanceOf[String])
    }
    if (map.containsKey(CIS.parent_id.name) && map.get(CIS.parent_id.name) != null) {
      setParentAsContainmentProperties(item, asCiPKType(map.get(CIS.parent_id.name)), depth)
    }
    setProperties(item, pk, depth, ciDataProvider)
    setChildrenAsContainmentProperties(item, pk, depth)
    handleSourceArtifact(item, pk)
    item
  }

  private def setParentAsContainmentProperties(item: ConfigurationItem, parentId: CiPKType, depth: Int): Unit = {
    def predicate(pd: PropertyDescriptor): Boolean = pd.isAsContainment && pd.getKind == PropertyKind.CI
    val filteredProps = listNonTransientProperties(item.getType).filter(predicate)
    logger.debug(s"setParentAsContainmentProperties - item(${item.getId}):::parentId($parentId):::filteredProps(${filteredProps.size})")
    if (filteredProps.nonEmpty){
      val parentCi = readByPk(parentId,depth)
      filteredProps.foreach( pd => pd.set(item,parentCi))
    }
  }

  private def setChildrenAsContainmentProperties(item: ConfigurationItem, pk: CiPKType, depth: Int): Unit = {
    def predicate(pd: PropertyDescriptor): Boolean = pd.isAsContainment && pd.getKind == PropertyKind.SET_OF_CI
    val filteredProps = listNonTransientProperties(item.getType).filter(predicate)
    logger.debug(s"setChildrenAsContainmentProperties - item(${item.getId}):::pk($pk):::filteredProps(${filteredProps.size})")
    if (filteredProps.nonEmpty) {
      val children = jdbcTemplate.query(SELECT_CI_BY_PARENT_ID, MapRowMapper, pk).asScala
      logger.debug(s"setChildrenAsContainmentProperties - item(${item.getId}):::pk($pk):::children(${children.size})")
      filteredProps.foreach(pd => {
        val filteredChildren = children.filter { map =>
          Type.valueOf(map.get(CIS.ci_type.name).asInstanceOf[String]).instanceOf(pd.getReferencedType)
        }.map(map => asCiPKType(map.get(CIS.ID.name)))
        logger.debug(s"setChildrenAsContainmentProperties - item(${item.getId}):::pk($pk):::pd($pd):::filteredChildren(${filteredChildren.size})")
        pd.set(item,
          if (filteredChildren.nonEmpty) new util.HashSet(readByPks(filteredChildren.toSeq, depth).asJava)
          else new util.HashSet()
        )
      })
    }
  }

  private def newCiAttributes(ciMap: util.Map[String, AnyRef]): CiAttributes =
    new CiAttributes(
      ciMap.get(CIS.created_by.name).asInstanceOf[String],
      ciMap.get(CIS.created_at.name).asInstanceOf[DateTime],
      ciMap.get(CIS.modified_by.name).asInstanceOf[String],
      ciMap.get(CIS.modified_at.name).asInstanceOf[DateTime],
      ciMap.get(CIS.scm_traceability_data_id.name).asInstanceOf[Integer]
    )

  private def readExternalProperties(values: util.List[util.Map[String, AnyRef]]): util.Map[String, ExternalProperty] = {
    import LOOKUP_VALUES._
    val result = new util.HashMap[String, ExternalProperty]()
    values.forEach { map =>
      val property = map.get(name.name).asInstanceOf[String]
      val lookupValueKey = new LookupValueKey
      lookupValueKey.setKey(map.get(key.name).asInstanceOf[String])
      lookupValueKey.setProviderId(readByPk(asCiPKType(map.get(provider.name)), 0).getId)
      result.put(property, lookupValueKey)
    }
    result
  }

  private def setProperties(item: ConfigurationItem, pk: CiPKType, depth: Int, ciDataProvider: CiDataProvider): Unit = {
    val ciFromMap: CiFromMapReader = (map, dataProvider) => readFromMap(map, depth, dataProvider)
    new GenericPropertiesReader(pk, item, passwordEncrypter, ciFromMap)
      .setGenericProperties(ciDataProvider.getProperties(pk))
    createTypeSpecificReaders(item.getType, pk)
      .foreach(_.readProperties(item, readByPk(_, depth), convert)(ciDataProvider, ciFromMap, typeSpecificPersisterFactories))
  }

  private def handleSourceArtifact(item: ConfigurationItem, pk: CiPKType): Unit = {
    item match {
      case artifact: SourceArtifact =>
        artifact.setFile(SourceArtifactFile.withNullableWorkDir(getFileName(artifact, pk), artifact, workDir))
      case _ =>
    }
  }

  private def getFileName(artifact: SourceArtifact, pk: CiPKType): String =
    Option(artifactRepository.getFilename(artifact.getId))
      .orElse(resolveAndStore(artifact, pk))
      .getOrElse(FilenameUtils.getName(artifact.getFileUri))

  private def resolveAndStore(artifact: SourceArtifact, pk: CiPKType): Option[String] =
    Option(artifact.getFileUri)
      .map(ArtifactResolverRegistry.lookup(_).resolveLocation(artifact))
      .map(using(_) { resolvedArtifact =>
        val fileName = resolvedArtifact.getFileName
        artifactRepository.updateFilename(pk, fileName)
        fileName
      })

  private def convert(pd: PropertyDescriptor, value: Any): Any = {
    pd.getKind match {
      case MAP_STRING_STRING =>
        convertMap(value.asInstanceOf[util.Map[String, String]], pd.isPassword, passwordEncrypter)
      case SET_OF_STRING =>
        convertSet(value.asInstanceOf[util.Set[String]], pd.isPassword, passwordEncrypter)
      case LIST_OF_STRING =>
        convertList(value.asInstanceOf[util.List[String]], pd.isPassword, passwordEncrypter)
      case PropertyKind.BOOLEAN => enforceBoolean(value)
      case PropertyKind.DATE => enforceDate(value)
      case PropertyKind.INTEGER => enforceInteger(value)
      case _ => value
    }
  }


  class GenericPropertiesReader(val pk: CiPKType,
                                val item: ConfigurationItem,
                                override val passwordEncrypter: PasswordEncrypter,
                                ciFromMap: (util.Map[String, Object], CiDataProvider) => ConfigurationItem)
    extends EncryptedPropertyReader {

    import PropertyKind._

    type ROW = util.Map[String, AnyRef]
    type PropValues[K, V] = mutable.Map[String, mutable.TreeMap[K, V]]

    private val listStringPropertyValues = mutable.Map[String, mutable.TreeMap[Int, String]]()
    private val setStringPropertyValues = mutable.Map[String, mutable.TreeMap[Int, String]]()
    private val listCiPropertyValues = mutable.Map[String, mutable.TreeMap[Int, ConfigurationItem]]()
    private val mapStringPropertyValues = mutable.Map[String, mutable.TreeMap[String, String]]()

    def setGenericProperties(properties: util.List[ROW]): Unit = {
      val descriptor = item.getType.getDescriptor
      val ciIdAndTypes = properties.asScala.flatMap { map =>
        Option(map.get(CI_PROPERTIES.ci_ref_value.name)).map(pk => asCiPKType(pk) -> map.get(s"a_${CIS.ci_type.name}").toString.ciType)
      }
      implicit val ciGroupDataProvider: CiDataProvider = new CiGroupCiDataProvider(ciIdAndTypes.toSeq)
      properties.forEach { row =>
        processPropertyValue(descriptor, row)
      }
      listCiPropertyValues.foreach { case (propertyName, ciPKs) =>
        setListCiPropertyValues(descriptor.getPropertyDescriptor(propertyName), ciPKs)
      }
      listStringPropertyValues.foreach { case (propertyName, values) =>
        val pd = descriptor.getPropertyDescriptor(propertyName)
        pd.set(item, convert(pd, toList(values)))
      }
      setStringPropertyValues.foreach { case (propertyName, values) =>
        val pd = descriptor.getPropertyDescriptor(propertyName)
        pd.set(item, convert(pd, toSet(values)))
      }
      mapStringPropertyValues.foreach { case (propertyName, values) =>
        val pd = descriptor.getPropertyDescriptor(propertyName)
        pd.set(item, convert(pd, values.asJava))
      }
    }

    private def extractCi(row: ROW)(implicit ciDataProvider: CiDataProvider): ConfigurationItem = {
      val ciRow: ROW = CIS.allFields.foldLeft(Map[String, AnyRef]()) { case (acc, field) =>
        acc + (field.name -> row.get(s"a_${field.name.toLowerCase}"))
      }.asJava
      ciFromMap(ciRow, ciDataProvider)
    }

    private def processPropertyValue(descriptor: Descriptor, row: ROW)(implicit ciDataProvider: CiDataProvider): Unit = {
      val propertyName = row.get(CI_PROPERTIES.name.name).asInstanceOf[String]
      val propertyDescriptor = descriptor.getPropertyDescriptor(propertyName)
      if (propertyDescriptor == null) {
        val pk = asCiPKType(row.get(CI_PROPERTIES.ci_id.name))
        logger.warn(s"Unknown property '$propertyName' found in database for known type '${descriptor.getType}' [CI_ID = $pk], ignoring.")
      } else {
        val stringVal: ROW => String = getStringValue(_, propertyDescriptor.isPassword)
        val rowId = asCiPKType(row.get(CI_PROPERTIES.ID.name))
        val propertyKind = propertyDescriptor.getKind
        propertyKind match {
          case STRING =>
            propertyDescriptor.set(item, schemaInfo.sqlDialect.enforceNotNull(stringVal(row)))
          case k if k.isSimple =>
            propertyDescriptor.set(item, getSimpleValue(k, row))
          case CI =>
            propertyDescriptor.set(item, extractCi(row))
          case LIST_OF_CI | SET_OF_CI =>
            setAndHandleDuplicates(listCiPropertyValues, propertyName, getIndexValue(row), extractCi(row), rowId)
          case LIST_OF_STRING =>
            setAndHandleDuplicates(listStringPropertyValues, propertyName, getIndexValue(row), stringVal(row), rowId)
          case SET_OF_STRING =>
            setAndHandleDuplicates(setStringPropertyValues, propertyName, getIndexValue(row), stringVal(row), rowId)
          case MAP_STRING_STRING =>
            setAndHandleDuplicates(mapStringPropertyValues, propertyName, getKeyValue(row), stringVal(row), rowId)
        }
      }
    }

    def setAndHandleDuplicates[K: Ordering, V](propValues: PropValues[K, V], propertyName: String, key: K, value: V, rowId: CiPKType): Unit = {
      // DEPL-13380 DEPL-13955 - Due to a bug we may encounter duplicate or conflicting entries for collection-valued CI
      // properties. We handle these here: detect whether a value was already defined earlier for this entry (i.e.
      // property+key) while reading this CI. If the current entry is a duplicate, we just delete the current row from
      // the DB. If it's a different value, we don't know what the right value should be so we just bail out.
      handle(propValues, propertyName, key, value) match {
        case None =>
        case Some(prevValue) =>
          if (value == prevValue)
            jdbcTemplate.update(DELETE_PROPERTY, rowId)
          else
            throw new IllegalStateException(
              // not mentioning the values in the errmsg since they are decrypted so they must not appear in the logs
              s"Table ${CI_PROPERTIES.tableName.name} row ID=$rowId defines conflicting value for propertyName $propertyName and key [$key] - a different value was encountered earlier."
            )
      }
    }

    private def handle[K: Ordering, V](propertyValues: PropValues[K, V], propertyName: String, idx: K, value: V): Option[V] = {
      propertyValues.getOrElseUpdate(propertyName, mutable.TreeMap[K, V]()).put(idx, value)
    }

    private def setListCiPropertyValues(propertyDescriptor: PropertyDescriptor, values: mutable.TreeMap[Int, ConfigurationItem]): Unit = {
      propertyDescriptor.getKind match {
        case SET_OF_CI => propertyDescriptor.set(item, toSet(values))
        case LIST_OF_CI => propertyDescriptor.set(item, toList(values))
        case _ => throw new java.lang.AssertionError("assertion failed: Not list property -> cannot happen!")
      }
    }

    private def toSet[T](values: collection.Map[Int, T]): util.Set[T] = new util.LinkedHashSet(values.values.toSet.asJava)

    private def toList[T](values: collection.Map[Int, T]): util.List[T] = new util.ArrayList(values.values.toList.asJava)

  }

}
