package com.xebialabs.deployit.repository.sql.properties

import java.util

import com.xebialabs.deployit.core.sql.batch.{BatchCommand, BatchCommandWithArgs, BatchCommandWithSetter}
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind, Type}
import com.xebialabs.deployit.repository.sql.base._

trait PropertyInserter {

  protected def ciPropertiesRepository: InsertCiPropertiesRepository

  protected def getValue(pd: PropertyDescriptor): Option[Any]

  protected val propertyFilter: PropertyDescriptor => Boolean = !_.isTransient

  @Deprecated
  def migrationInsertProperties(t: Type, ciPk: CiPKType, handler: PropertyErrorHandler = defaultErrorHandler): Unit = {
    val typeSpecificInserters = ciPropertiesRepository.createTypeSpecificInserters(t, ciPk)
    forAllNonTransientProperties(t, propertyFilter) { pd =>
      getValue(pd).foreach { value =>
        if (typeSpecificInserters.isEmpty || typeSpecificInserters.forall(persister => !persister.insertProperty(pd.getName, value))) {
          migrationInsertPropertiesGeneric(ciPk, pd, value, handler)
        }
      }
    }
    typeSpecificInserters.foreach(_.finish(handler))
  }

  private def migrationInsertPropertiesGeneric(ciPk: CiPKType, pd: PropertyDescriptor, value: Any, handler: PropertyErrorHandler):Unit = {
    val propertyName = pd.getName
    pd.getKind match {
      case PropertyKind.CI if pd.isAsContainment  => // do not persist
      case PropertyKind.SET_OF_CI if pd.isAsContainment  => // do not persist
      case k if k.isSimple || k == PropertyKind.CI =>
        ciPropertiesRepository.insertProperty(ciPk, propertyName, pd.getKind, value, handler)
      case PropertyKind.LIST_OF_CI | PropertyKind.SET_OF_CI | PropertyKind.LIST_OF_STRING | PropertyKind.SET_OF_STRING =>
        ciPropertiesRepository.insertIndexedProperty(ciPk, propertyName, pd.getKind, value.asInstanceOf[util.Collection[String]], handler = handler)
      case PropertyKind.MAP_STRING_STRING =>
        ciPropertiesRepository.insertKeyedProperty(ciPk, propertyName, pd.getKind, value.asInstanceOf[util.Map[String, String]], handler)
    }
  }

  def insertProperties(t: Type, ciPk: CiPKType): (Seq[BatchCommandWithArgs], Seq[BatchCommandWithSetter]) = {
    val typeSpecificInserters = ciPropertiesRepository.createTypeSpecificInserters(t, ciPk)

    val allGenericCommands: Seq[BatchCommandWithArgs] = listNonTransientProperties(t, propertyFilter)
      .flatMap { pd =>
        getValue(pd).map { value =>
          if (typeSpecificInserters.isEmpty || typeSpecificInserters.forall(persister => !persister.insertProperty(pd.getName, value))) {
            insertPropertiesGeneric(ciPk, pd, value)
          } else
            Vector.empty
        }.getOrElse(Vector.empty)
      }

    (allGenericCommands, typeSpecificInserters.flatMap(_.batchFinish()))
  }

  private def insertPropertiesGeneric(ciPk: CiPKType, pd: PropertyDescriptor, value: Any): Vector[BatchCommandWithArgs] = {
    val propertyName = pd.getName
    pd.getKind match {
      case PropertyKind.CI if pd.isAsContainment  => Vector.empty // do not persist
      case PropertyKind.SET_OF_CI if pd.isAsContainment  => Vector.empty // do not persist
      case k if k.isSimple || k == PropertyKind.CI =>
        Vector(ciPropertiesRepository.batchInsertProperty(ciPk, propertyName, pd.getKind, value))
      case PropertyKind.LIST_OF_CI | PropertyKind.SET_OF_CI | PropertyKind.LIST_OF_STRING | PropertyKind.SET_OF_STRING =>
        ciPropertiesRepository.batchInsertIndexedProperty(ciPk, propertyName, pd.getKind, value.asInstanceOf[util.Collection[String]]).toVector
      case PropertyKind.MAP_STRING_STRING =>
        ciPropertiesRepository.batchInsertKeyedProperty(ciPk, propertyName, pd.getKind, value.asInstanceOf[util.Map[String, String]]).toVector
    }
  }
}
