package com.xebialabs.deployit.repository.sql.lock

import com.xebialabs.deployit.core.sql.ColumnName
import com.xebialabs.deployit.core.sql.Queries
import com.xebialabs.deployit.core.sql.SchemaInfo
import com.xebialabs.deployit.core.sql.TableName
import com.xebialabs.deployit.engine.tasker.TaskId
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.sql.lock.model.CiLock
import com.xebialabs.deployit.sql.base.schema.CIS
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.dao.DuplicateKeyException
import org.springframework.dao.EmptyResultDataAccessException
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.jdbc.core.RowMapper
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional

import java.sql.ResultSet
import java.util.Date
import java.util.{Set => JSet}
import scala.jdk.CollectionConverters._

trait CiLockRepository {
  def listAllLocks(): List[String]

  def listLocksByTaskId(taskId: TaskId): List[String]

  def clearLocksByTaskId(taskId: TaskId): Unit

  def clearAllLocks(): Unit

  def lock(ci: ConfigurationItem, taskId: TaskId): Boolean

  def lock(cis: JSet[ConfigurationItem], taskId: TaskId): Boolean

  def unlock(cis: JSet[ConfigurationItem], taskId: TaskId): Unit
}

@Component
@Transactional("mainTransactionManager")
class CiLockRepositoryImpl(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate)
                          (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends CiLockRepository with CiLockQueries with Logging {

  override def listLocksByTaskId(taskId: TaskId): List[String] =
    jdbcTemplate.query(LIST_LOCKS_BY_TASK_ID, (rs: ResultSet, _: Int) => rs.getString(1), taskId).asScala.toList

  override def listAllLocks(): List[String] =
    jdbcTemplate.query(LIST_ALL_LOCKS, (rs: ResultSet, _: Int) => rs.getString(1)).asScala.toList

  override def clearLocksByTaskId(taskId: String): Unit =
    jdbcTemplate.update(CLEAR_LOCKS_WITH_TASK_ID, taskId)

  override def clearAllLocks(): Unit =
    jdbcTemplate.update(CLEAR_ALL_LOCKS)

  def getLock(ci: ConfigurationItem): Option[CiLock] =
    try {
      Option(jdbcTemplate.queryForObject(GET_LOCKS_BY_CI_ID, new RowMapper[CiLock]() {
        override def mapRow(rs: ResultSet, rowNum: Int): CiLock =
          CiLock(rs.getInt(1), rs.getString(2))
      }, ci.get$internalId()))
    } catch {
      case _: EmptyResultDataAccessException =>
        Option.empty[CiLock]
    }

  def lock(ci: ConfigurationItem, taskId: TaskId): Boolean = {
    val existingCiLock: Option[CiLock] = getLock(ci)

    if (existingCiLock.nonEmpty) {
      return canReentrant(ci, taskId, existingCiLock.get)
    }

    try {
      val res = jdbcTemplate.update(LOCK, ci.get$internalId(), taskId)
      logger.info(s"Locking ${ci.getName} on ${new Date()}")
      if (res == 0) {
        logger.info(s"${ci.getName} already has a lock")
      }
      res > 0
    } catch {
      case _ : DuplicateKeyException =>
        logger.warn(s"${ci.getName} is locked by another task  : ${getLock(ci).get.taskId}")
        false
      case _: Exception =>
        false
    }
  }

  private def canReentrant(ci: ConfigurationItem, taskId: TaskId, existingLock: CiLock): Boolean = {
    if (existingLock.taskId.equals(taskId)) {
      logger.info(s"Current task ${taskId} has already locked ${ci.getName}")
      true
    } else {
      logger.warn(s"${ci.getName} is locked by another task  : ${existingLock.taskId}")
      false
    }
  }

  override def lock(cis: JSet[ConfigurationItem], taskId: TaskId): Boolean = cis.asScala.forall(ci => lock(ci, taskId))

  override def unlock(cis: JSet[ConfigurationItem], taskId: TaskId): Unit =
    cis.asScala.foreach(ci => {
      jdbcTemplate.update(UNLOCK, ci.get$internalId(), taskId)
    })
}

object CiLockSchema {
  val tableName: TableName = TableName("XLD_CI_LOCK")
  val ci_id: ColumnName = ColumnName("CI_ID")
  val task_id: ColumnName = ColumnName("task_id")
}

trait CiLockQueries extends Queries {

  import CiLockSchema._

  val LIST_LOCKS_BY_TASK_ID = sqlb"select c.${CIS.path} from $tableName t inner join ${CIS.tableName} c on t.$ci_id = c.${CIS.ID} where $task_id = ?"
  val LIST_ALL_LOCKS = sqlb"select c.${CIS.path} from $tableName t inner join ${CIS.tableName} c on t.$ci_id = c.${CIS.ID}"

  val CLEAR_LOCKS_WITH_TASK_ID = sqlb"delete from $tableName where $task_id = ?"
  val CLEAR_ALL_LOCKS = sqlb"delete from $tableName"

  val LOCK = sqlb"insert into $tableName ($ci_id, $task_id) values (?, ?)"
  val GET_LOCKS_BY_CI_ID = sqlb"select $ci_id, $task_id from $tableName where $ci_id = ?"

  val UNLOCK = sqlb"delete from $tableName where $ci_id = ? and $task_id = ?"
}

