package com.xebialabs.deployit.repository.sql

import com.xebialabs.deployit.checksum.ChecksumAlgorithmProvider
import com.xebialabs.deployit.core.sql.batch.BatchExecutorRepository
import com.xebialabs.deployit.core.sql.spring.{MapRowMapper, Setter, toMap, transactional}
import com.xebialabs.deployit.core.sql.util.queryWithInClause
import com.xebialabs.deployit.core.sql.{SqlCondition => cond}
import com.xebialabs.deployit.core.sql._
import com.xebialabs.deployit.event.EventBusHolder
import com.xebialabs.deployit.plugin.api.reflect._
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository._
import com.xebialabs.deployit.repository.sql.artifacts.ArtifactDataRepository
import com.xebialabs.deployit.repository.sql.artifacts.ArtifactRepository
import com.xebialabs.deployit.repository.sql.base._
import com.xebialabs.deployit.repository.sql.commands.util.CommandConverter
import com.xebialabs.deployit.repository.sql.commands.CreateCommand
import com.xebialabs.deployit.repository.sql.commands._
import com.xebialabs.deployit.repository.sql.reader.CiReader
import com.xebialabs.deployit.repository.sql.reader.CiReaderContext
import com.xebialabs.deployit.repository.sql.specific._
import com.xebialabs.deployit.repository.validation.CommandValidator
import com.xebialabs.deployit.security.sql.SqlPermissionService
import com.xebialabs.deployit.sql.base.schema.{CIS, CI_PROPERTIES}
import com.xebialabs.deployit.task.archive.TaskArchiveStore
import com.xebialabs.deployit.util.NullPasswordEncrypter
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.license.LicenseCiCounter
import com.xebialabs.license.service.LicenseTransaction
import com.xebialabs.xlplatform.coc.service.SCMTraceabilityService
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.annotation.DependsOn
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.jdbc.core.RowMapper
import org.springframework.jdbc.core.SingleColumnRowMapper
import org.springframework.stereotype.Repository
import org.springframework.transaction.PlatformTransactionManager
import org.springframework.transaction.annotation.Transactional

import java.sql.ResultSet
import java.util.{ArrayList => JArrayList, List => JList, Map => JavaMap, Set => JSet}
import ai.digital.deploy.core.notification.api.EventNotificationService
import com.xebialabs.deployit.exception.NotFoundException

import scala.collection.mutable
import scala.jdk.CollectionConverters._

case class CiPkAndType(pk: CiPKType, ciType: Type)

object CiPkAndType {

  def apply(configurationItem: ConfigurationItem): CiPkAndType =
    CiPkAndType(configurationItem.get$internalId, configurationItem.getType)

  def apply(m: JavaMap[String, Object]): CiPkAndType =
    CiPkAndType(asCiPKType(m.get(CIS.ID.name)), Type.valueOf(m.get(CIS.ci_type.name).asInstanceOf[String]))
}

trait CiRepository {

  def execute(changeSet: ChangeSet, progressLogger: ProgressLogger, licenseCiCounter: LicenseCiCounter, licenseTransaction: LicenseTransaction): Unit

  def readByReferenceId[T <: ConfigurationItem](referenceId: String, workDir: WorkDir, depth: Int, useCache: Boolean, decryptPasswords: Boolean): T

  def read[T <: ConfigurationItem](id: String, workDir: WorkDir, depth: Int, useCache: Boolean = true, decryptPasswords: Boolean = true): T

  def read[T <: ConfigurationItem](ids: JList[String], depth: Int, useCache: Boolean, decryptPasswords: Boolean, skipNotExistingCis: Boolean): JList[T]

  def read[T <: ConfigurationItem](ids: JList[String], workDir: WorkDir, depth: Int, useCache: Boolean, decryptPasswords: Boolean, skipNotExistingCis: Boolean): JList[T]

  def exists(item: ConfigurationItem): Boolean

  def exists(id: String): Boolean

  def selectExistingPaths(ids: JSet[String]): JSet[String]

  def list(criteria: SearchParameters): JList[ConfigurationItemData]

  def listInternal(criteria: SearchParameters): JList[ConfigurationItemDataWithInternalId]

  def listEntities[T <: ConfigurationItem](criteria: SearchParameters): JList[T]

  def listEntities[T <: ConfigurationItem](criteria: SearchParameters, depth: Int, useCache: Boolean): JList[T]

  def count(criteria: SearchParameters): Integer

  def validate(changeSet: ChangeSet, licenseCiCounter: LicenseCiCounter, licenseTransaction: LicenseTransaction): Unit

  def findReferencesTo(id: String, searchParameters: SearchParameters): JList[ConfigurationItemData]

  def findCiIdAndTypeFromPaths(paths: Seq[String]): List[JavaMap[String, Object]]
}

@Repository
@DependsOn(Array("configurationHolderInitilizer"))
class CiRepositoryImpl(@Autowired(required = false) val typeSpecificPersisterFactories: JList[TypeSpecificPersisterFactory],
                       @Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                       @Autowired @Qualifier("mainTransactionManager") transactionManager: PlatformTransactionManager,
                       @Autowired val ciHistoryRepository: CiHistoryRepository,
                       @Autowired val commandValidator: CommandValidator,
                       @Autowired val passwordEncrypter: PasswordEncrypter,
                       @Autowired val artifactRepository: ArtifactRepository,
                       @Autowired val artifactDataRepository: ArtifactDataRepository,
                       @Autowired val sqlPermissionService: SqlPermissionService,
                       @Autowired val scmTraceabilityService: SCMTraceabilityService,
                       @Autowired val checksumAlgorithmProvider: ChecksumAlgorithmProvider,
                       @Autowired val taskArchiveStore: TaskArchiveStore,
                       @Autowired val lifecycleHooks: JList[CiLifecycleHook],
                       @Autowired val eventNotificationService: EventNotificationService,
                       @Autowired @Qualifier("mainBatchExecutorRepository") val batchExecutorRepository: BatchExecutorRepository)
                      (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends CiRepository with CiExists with CiQueries {
  private implicit val factories: JList[TypeSpecificPersisterFactory] = typeSpecificPersisterFactories

  private def createTypeSpecificReaders(t: Type, pk: CiPKType): List[TypeSpecificReader] =
    typeSpecificPersisterFactories.asScala.flatMap(_.createReader(t, pk)).toList

  private def createTypeSpecificInserters(t: Type, pk: CiPKType): List[TypeSpecificInserter] =
    typeSpecificPersisterFactories.asScala.flatMap(_.createInserter(t, pk)).toList

  private def createTypeSpecificUpdaters(t: Type, pk: CiPKType): List[TypeSpecificUpdater] =
    typeSpecificPersisterFactories.asScala.flatMap(_.createUpdater(t, pk)).toList

  private def createTypeSpecificDeleters(t: Type, pk: CiPKType): List[TypeSpecificDeleter] =
    typeSpecificPersisterFactories.asScala.flatMap(_.createDeleter(t, pk)).toList

  private def createTypeSpecificReferenceFinder(pks: CiPKType*): List[TypeSpecificReferenceFinder] =
    typeSpecificPersisterFactories.asScala.map(_.createReferenceFinder(pks)).toList

  def execute(changeSet: ChangeSet, progressLogger: ProgressLogger, licenseCiCounter: LicenseCiCounter,
              licenseTransaction: LicenseTransaction): Unit = {
    val context = new ChangeSetContext(licenseTransaction, progressLogger, Option(changeSet.getSCMTraceabilityData))
    val steps = transactional(transactionManager, readOnly = true) {
      val allSteps = if (changeSet.isForceRedeploy()) changeSetStepsRedeploy(changeSet, licenseCiCounter) else changeSetSteps(changeSet, licenseCiCounter)
      allSteps.foreach(_.preloadCache(context))
      allSteps
    }
    transactional(transactionManager) {
      steps.foreach(_.execute(context))
    }
    progressLogger.log("Done")
    eventNotificationService.sendChangeSetEvent(CommandConverter.toChangeSetEvent(steps))
    EventBusHolder.publish(CommandConverter.toChangeSetEvent(steps))
  }

  private def combineAndSort(set1: Seq[ConfigurationItem], set2: Seq[ConfigurationItem]) =
    (set1 ++ set2).sortBy(_.getId.count(_ equals '/'))

  override def readByReferenceId[T <: ConfigurationItem](referenceId: String, workDir: WorkDir, depth: Int, useCache: Boolean, decryptPasswords: Boolean): T = {
    try {
      val criteria = new SearchParameters().setReferenceId(referenceId)
      val ciData = list(criteria).asScala.head
      read(ciData.getId, workDir, depth, useCache, decryptPasswords)
    } catch {
      case _: Exception => throw new NotFoundException("Repository entity with reference id [%s] not found", referenceId)
    }
  }

  override def read[T <: ConfigurationItem](id: String, workDir: WorkDir, depth: Int, useCache: Boolean = true, decryptPasswords: Boolean = true): T =
    createReader(workDir, getCache(useCache), decryptPasswords).readByPath(idToPath(id), depth).asInstanceOf[T]

  @Transactional("mainTransactionManager")
  override def read[T <: ConfigurationItem](ids: JList[String], depth: Int, useCache: Boolean, decryptPasswords: Boolean, skipNotExistingCis: Boolean): JList[T] =
    read(ids, null, depth, useCache, decryptPasswords, skipNotExistingCis)

  @Transactional("mainTransactionManager")
  override def read[T <: ConfigurationItem](ids: JList[String], workDir: WorkDir, depth: Int, useCache: Boolean, decryptPasswords: Boolean, skipNotExistingCis: Boolean): JList[T] = {
    val reader = createReader(workDir, getCache(useCache), decryptPasswords)
    new JArrayList(reader.readByPaths(ids.asScala.map(idToPath).toSeq, depth, useCache, skipNotExistingCis).asJava.asInstanceOf[JList[T]])
  }

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def exists(item: ConfigurationItem): Boolean = exists(item.getId)

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def selectExistingPaths(paths: JSet[String]): JSet[String] = selectExistingPaths(paths.asScala.toSeq).asJava

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def list(criteria: SearchParameters): JList[ConfigurationItemData] = {
    val queryBuilder: QueryBuilder = applySecurityParameters(new SearchParametersSelectBuilder(criteria).selectBuilder, criteria)
    jdbcTemplate.query(queryBuilder.query, Setter(queryBuilder.parameters), new RowMapper[ConfigurationItemData] {
      override def mapRow(rs: ResultSet, rowNum: Int): ConfigurationItemData =
        new ConfigurationItemData(pathToId(rs.getString(1)), Type.valueOf(rs.getString(2)))
    })
  }

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def listInternal(criteria: SearchParameters): JList[ConfigurationItemDataWithInternalId] = {
    val queryBuilder: QueryBuilder = applySecurityParameters(new SearchParametersExtendedSelectBuilder(criteria).selectBuilder, criteria)
    jdbcTemplate.query(queryBuilder.query, Setter(queryBuilder.parameters), new RowMapper[ConfigurationItemDataWithInternalId] {
      override def mapRow(rs: ResultSet, rowNum: Int): ConfigurationItemDataWithInternalId =
        ConfigurationItemDataWithInternalId(pathToId(rs.getString(1)), rs.getString(2), rs.getInt(3), Type.valueOf(rs.getString(4)))
    })
  }

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def listEntities[T <: ConfigurationItem](criteria: SearchParameters): JList[T] = listEntities(criteria, Integer.MAX_VALUE, useCache = true)

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def listEntities[T <: ConfigurationItem](criteria: SearchParameters, depth: Int, useCache: Boolean): JList[T] = {
    val queryBuilder: QueryBuilder = applySecurityParameters(new SearchParametersFullSelectBuilder(criteria).selectBuilder, criteria)
    val ciReader = createReader(null, getCache(useCache))
    jdbcTemplate.query(queryBuilder.query, Setter(queryBuilder.parameters), new RowMapper[T] {
      override def mapRow(rs: ResultSet, rowNum: Int): T = {
        val map = toMap(rs)
        val pk = asCiPKType(map.get(CIS.ID.name))
        ciReader.readUsingCache(pk, depth)(map).asInstanceOf[T]
      }
    })
  }

  private def applySecurityParameters(selectBuilder: SelectBuilder, criteria: SearchParameters): QueryBuilder = {
    if (criteria.hasSecurityParameters) {
      new WithPermissionsBuilder(selectBuilder, criteria.getRoles, criteria.getPermissions).selectBuilder
    } else {
      selectBuilder
    }
  }

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def count(criteria: SearchParameters): Integer = {
    val selectBuilder = applySecurityParameters(new SearchParametersCountBuilder(criteria).selectBuilder, criteria)
    jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), new SingleColumnRowMapper(classOf[Integer])).asScala.headOption.getOrElse(0)
  }

  private def createReader(
                            workDir: WorkDir,
                            cache: mutable.HashMap[CiPKType, ConfigurationItem] = CiReaderContext.get.cache,
                            decryptPasswords: Boolean = true
                          ): CiReader = {
    val passEncrypter = if (decryptPasswords) passwordEncrypter else NullPasswordEncrypter.getInstance()
    new CiReader(workDir, cache, jdbcTemplate, passEncrypter, artifactRepository, createTypeSpecificReaders)
  }

  private def getCache(useCache: Boolean) =
    if (useCache) {
      CiReaderContext.get.cache
    } else {
      new mutable.HashMap[CiPKType, ConfigurationItem]()
    }

  @Transactional("mainTransactionManager")
  override def validate(changeSet: ChangeSet, licenseCiCounter: LicenseCiCounter, licenseTransaction: LicenseTransaction): Unit = {
    val context = new ChangeSetContext(licenseTransaction, new NullProgressLogger)
    if (changeSet.isForceRedeploy()) {
      changeSetStepsRedeploy(changeSet, licenseCiCounter).foreach(_.validate(context))
    } else {
      changeSetSteps(changeSet, licenseCiCounter).foreach(_.validate(context))
    }
  }

  private def changeSetSteps(changeSet: ChangeSet, licenseCiCounter: LicenseCiCounter): List[ChangeSetCommand] = {
    val (toUpdate, toCreate) = changeSet.getCreateOrUpdateCis.asScala.partition(exists)
    new CreateCommand(jdbcTemplate, artifactRepository, artifactDataRepository, this, ciHistoryRepository, batchExecutorRepository,
      passwordEncrypter, licenseCiCounter, createTypeSpecificInserters, scmTraceabilityService,
      combineAndSort(changeSet.getCreateCis.asScala.toSeq, toCreate.toSeq), checksumAlgorithmProvider, taskArchiveStore) ::
      new UpdateCommand(jdbcTemplate, artifactRepository, artifactDataRepository, this, ciHistoryRepository, batchExecutorRepository,
        passwordEncrypter, createTypeSpecificUpdaters, scmTraceabilityService,
        combineAndSort(changeSet.getUpdateCis.asScala.toSeq, toUpdate.toSeq), checksumAlgorithmProvider) ::
      new MoveCommand(jdbcTemplate, artifactDataRepository, commandValidator, taskArchiveStore, changeSet.getMoveCis.asScala) ::
      new RenameCommand(jdbcTemplate, artifactDataRepository, changeSet.getRenameCis.asScala) ::
      new CopyCommand(jdbcTemplate, artifactRepository, artifactDataRepository, this, ciHistoryRepository, batchExecutorRepository,
        passwordEncrypter, licenseCiCounter, createReader(null), createTypeSpecificInserters, commandValidator, scmTraceabilityService,
        changeSet.getCopyCis.asScala, checksumAlgorithmProvider, taskArchiveStore) ::
      new DeleteCommand(jdbcTemplate, artifactDataRepository, this, ciHistoryRepository, licenseCiCounter,
        createTypeSpecificDeleters, createTypeSpecificReferenceFinder, changeSet.getDeleteCiIdsWithInternalIds.asScala, sqlPermissionService,
        lifecycleHooks.asScala.toList) ::
      Nil
  }

  private def changeSetStepsRedeploy(changeSet: ChangeSet, licenseCiCounter: LicenseCiCounter): List[ChangeSetCommand] = {
      val (toUpdate, toCreate) = changeSet.getCreateOrUpdateCis.asScala.partition(exists)
      new DeleteCommand(jdbcTemplate, artifactDataRepository, this, ciHistoryRepository, licenseCiCounter,
          createTypeSpecificDeleters, createTypeSpecificReferenceFinder, changeSet.getDeleteCiIdsWithInternalIds.asScala, sqlPermissionService,
          lifecycleHooks.asScala.toList) ::
        new CreateCommand(jdbcTemplate, artifactRepository, artifactDataRepository, this, ciHistoryRepository, batchExecutorRepository,
            passwordEncrypter, licenseCiCounter, createTypeSpecificInserters, scmTraceabilityService,
            combineAndSort(changeSet.getCreateCis.asScala.toSeq, toCreate.toSeq), checksumAlgorithmProvider, taskArchiveStore) ::
          new UpdateCommand(jdbcTemplate, artifactRepository, artifactDataRepository, this, ciHistoryRepository, batchExecutorRepository,
              passwordEncrypter, createTypeSpecificUpdaters, scmTraceabilityService,
              combineAndSort(changeSet.getUpdateCis.asScala.toSeq, toUpdate.toSeq), checksumAlgorithmProvider) ::
          new MoveCommand(jdbcTemplate, artifactDataRepository, commandValidator, taskArchiveStore, changeSet.getMoveCis.asScala) ::
          new RenameCommand(jdbcTemplate, artifactDataRepository, changeSet.getRenameCis.asScala) ::
          new CopyCommand(jdbcTemplate, artifactRepository, artifactDataRepository, this, ciHistoryRepository, batchExecutorRepository,
              passwordEncrypter, licenseCiCounter, createReader(null), createTypeSpecificInserters, commandValidator, scmTraceabilityService,
              changeSet.getCopyCis.asScala, checksumAlgorithmProvider, taskArchiveStore) ::
          Nil
    }


  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def findReferencesTo(id: String, searchParameters: SearchParameters): JList[ConfigurationItemData] = {
    val pk = jdbcTemplate.queryForObject(SELECT_ID_BY_PATH, classOf[CiPKType], idToPath(id))

    val selectBuilder = new SearchParametersSelectBuilder(searchParameters).selectBuilder

    var subselects = Seq(cond.subselect(CIS.ID,
      new SelectBuilder(CI_PROPERTIES.tableName).select(CI_PROPERTIES.ci_id).where(cond.equals(CI_PROPERTIES.ci_ref_value, pk))
    ))
    subselects = subselects ++ createTypeSpecificReferenceFinder(pk).flatMap(_.findReferencesBuilders()).map {
      cond.subselect(CIS.ID, _)
    }
    selectBuilder.where(cond.or(subselects))

    jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), new RowMapper[ConfigurationItemData] {
      override def mapRow(rs: ResultSet, rowNum: Int): ConfigurationItemData =
        new ConfigurationItemData(pathToId(rs.getString(1)), Type.valueOf(rs.getString(2)))
    })
  }

    class CisSearchPathsQueryBuilder(private val paths: Seq[String])
                                  (implicit schemaInfo: SchemaInfo) {

    val selectBuilder: SelectBuilder = new SelectBuilder(CIS.tableName)
      .select(CIS.ID)
      .select(CIS.ci_type)
      .select(CIS.path)
      .where(cond.in(CIS.path, paths))
  }

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def findCiIdAndTypeFromPaths(paths: Seq[String]): List[JavaMap[String, Object]] =
    queryWithInClause(paths.distinct) { idGroup =>
      val selectBuilder = new CisSearchPathsQueryBuilder(idGroup).selectBuilder
      jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), MapRowMapper).asScala.toList
    }
}
