package com.xebialabs.deployit.repository.sql

import com.xebialabs.deployit.core.sql.spring.{MapRowMapper, Setter, toMap}
import com.xebialabs.deployit.core.sql.{SqlCondition => cond, _}
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.sql.base.{CiPKType, _}
import com.xebialabs.deployit.security.sql.{CiResolver, ResolvedCi}
import com.xebialabs.deployit.sql.base.schema.CIS
import com.xebialabs.deployit.task.archive.TaskArchiveStore
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository
import org.springframework.transaction.annotation.Transactional

import java.sql.ResultSet
import scala.collection.mutable
import scala.jdk.CollectionConverters._

@Repository
class CiResolverImpl(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                     @Autowired taskArchiveStore: TaskArchiveStore)
                    (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends CiResolver with CiResolverQueries with SecuredCi with DirectoryRef {

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def getPkFromId(id: String): Number =
    jdbcTemplate.query(SELECT_CI_BY_PATH, MapRowMapper, idToPath(id)).asScala.headOption
      .map(m => asCiPKType(m.get(CIS.ID.name)))
      .getOrElse(throw new NotFoundException("Couldn't find ConfigurationItem [%s]", id))

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def getResolvedCiFromId(id: String): ResolvedCi =
    jdbcTemplate.query(SELECT_CI_BY_PATH, MapRowMapper, idToPath(id)).asScala.headOption
      .map(map =>
        ResolvedCi(
          asCiPKType(map.get(CIS.ID.name)),
          map.get(CIS.path.name).asInstanceOf[String],
          Type.valueOf(map.get(CIS.ci_type.name).asInstanceOf[String]),
          Option(map.get(CIS.parent_id.name)).map(asCiPKType),
          Option(map.get(CIS.secured_ci.name)).map(asCiPKType).orElse(Some(0)),
        )
      )
      .getOrElse(throw new NotFoundException("Couldn't find ConfigurationItem [%s]", id))

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def getIdFromPk(childPk: Number): String = pathToId(
    jdbcTemplate.queryForObject(SELECT_CI_BY_ID, MapRowMapper, asCiPKType(childPk)).get(CIS.path.name).asInstanceOf[String]
  )

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def getSecuredCiId(id: String): Option[CiPKType] = {
    def doGetSecuredCiId(path: String): Option[CiPKType] = {
      val builder = new SelectBuilder(CIS.tableName).select(CIS.secured_ci).where(cond.equals(CIS.path, path))
      jdbcTemplate.query(builder.query, Setter(builder.parameters),
        (rs: ResultSet, _: Int) => Integer.valueOf(rs.getInt(1))
      ).asScala.headOption
    }

    doGetSecuredCiId(idToPath(id)).orElse(parentPath(id).flatMap(doGetSecuredCiId))
  }

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def getSecuredCiId(pk: Number): Option[Number] = super.getSecuredCi(asCiPKType(pk))

  @Transactional(transactionManager = "mainTransactionManager")
  override def setSecuredCiId(resolvedCi: ResolvedCi): Unit =
    resolvedCi.securedCiPk match {
      case Some(securedCi) if securedCi == resolvedCi.pk => // Nothing, no change
      case Some(securedCi) =>
        jdbcTemplate.update(
          UPDATE_SECURED_CI_FOR_PATH,
          resolvedCi.pk, resolvedCi.path, s"${resolvedCi.path}/%", securedCi
        )

        updateArchivedSecureCis(resolvedCi.path, asCiPKType(resolvedCi.pk))
      case None =>
        jdbcTemplate.update(
          UPDATE_SECURED_CI_FOR_PATH_AND_NULL_SECURED_CI,
          resolvedCi.pk, resolvedCi.path, s"${resolvedCi.path}/%"
        )

        updateArchivedSecureCis(resolvedCi.path, asCiPKType(resolvedCi.pk))
    }

  @Transactional(transactionManager = "mainTransactionManager")
  override def removeSecuredCi(resolvedCi: ResolvedCi, parentSecuredCiPkOption: Option[Number]): Unit =
    resolvedCi.securedCiPk match {
      case Some(securedCi) if securedCi == resolvedCi.pk && resolvedCi.parentPk.isDefined =>
        parentSecuredCiPkOption
          .foreach { parentSecuredCi =>
            jdbcTemplate.update(
              UPDATE_SECURED_CI_FOR_PATH,
              parentSecuredCi, resolvedCi.path, s"${resolvedCi.path}/%", securedCi
            )
            updateArchivedSecureCis(resolvedCi.path, parentSecuredCi.intValue())
          }
      case Some(securedCi) if securedCi == resolvedCi.pk =>
        jdbcTemplate.update(
          UPDATE_SECURED_CI_FOR_PATH,
          null, resolvedCi.path, s"${resolvedCi.path}/%", securedCi
        )
        updateArchivedSecureCis(resolvedCi.path, securedCi = 0) // = null yields SQL error. CI 0 doesn't exist, but it should work anyway
      case Some(_) => // Do nothing
      case None => // Do nothing
    }

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def getPksFromIds(ids: Iterable[String]): Map[String, Integer] = {
    if (ids.isEmpty) {
      return Map()
    }

    val paths = ids.map(id => s"/$id")
    val builder = new SelectBuilder(CIS.tableName).select(CIS.path).select(CIS.ID).where(cond.in(CIS.path, paths))
    jdbcTemplate.query(builder.query, Setter(builder.parameters), new RowMapper[(String, Integer)] {
      override def mapRow(rs: ResultSet, rowNum: Int): (String, Integer) = {
        val map = toMap(rs)
        val pk = asCiPKType(map.get(CIS.ID.name))
        val id = pathToId(map.get(CIS.path.name).toString)
        (id, pk)
      }
    }).asScala.toMap
  }

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def getIdsFromPks(pks: Iterable[Number]): Map[Integer, String] = {
    if (pks.isEmpty) {
      return Map()
    }

    val builder = new SelectBuilder(CIS.tableName).select(CIS.ID).select(CIS.path).where(cond.in(CIS.ID, pks))
    jdbcTemplate.query(builder.query, Setter(builder.parameters), new RowMapper[(Integer, String)] {
      override def mapRow(rs: ResultSet, rowNum: Int): (Integer, String) = {
        val map = toMap(rs)
        val pk = asCiPKType(map.get(CIS.ID.name))
        val id = pathToId(map.get(CIS.path.name).toString)
        (pk, id)
      }
    }).asScala.toMap
  }

  private def updateArchivedSecureCis(pathString: String, securedCi: Integer): Unit = {
    val appType = "udm.Application"
    val envType = "udm.Environment"
    val cisInCondition = cond.in(CIS.ci_type, Seq(appType, envType))
    val appEnvsSecureCiMap =
      getTypeSecuredCiMapForSecuredCiAndPath(pathString, securedCi, cisInCondition)
        .groupBy { case (ciType, _) => ciType }
        .view
        .mapValues(_.map { case (_, pk) => pk }.toList)
        .toMap

    val otherTypesSecureCiMap =
      getTypeSecuredCiMapForSecuredCiAndPath(pathString, securedCi, cond.not(cisInCondition))
        .map { case (_, pk) => pk }
        .toList

    taskArchiveStore.updateSecureCi(securedCi,
      appEnvsSecureCiMap.get(Type.valueOf(envType)),
      appEnvsSecureCiMap.get(Type.valueOf(appType)),
      Option(otherTypesSecureCiMap))
  }

  private def getTypeSecuredCiMapForSecuredCiAndPath(pathString: String, securedCi: Number, condition: cond): mutable.Buffer[(Type, CiPKType)] = {
    val builder = new SelectBuilder(CIS.tableName).select(CIS.ID).select(CIS.ci_type)
      .where(condition)
      .where(cond.or(Seq(cond.equals(CIS.path, pathString),
        cond.and(Seq(cond.like(CIS.path, s"$pathString/%"), cond.equals(CIS.secured_ci, securedCi))))))
    jdbcTemplate.query(builder.query, Setter(builder.parameters), new RowMapper[(Type, Integer)] {
      override def mapRow(rs: ResultSet, rowNum: Int): (Type, Integer) = {
        val map = toMap(rs)
        val pk = asCiPKType(map.get(CIS.ID.name))
        val ciType = Type.valueOf(map.get(CIS.ci_type.name).toString)
        (ciType, pk)
      }
    }).asScala
  }

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def resolveDirectoryReference(path: String): String = resolveDirectoryRef(asCiPKType(getPkFromId(path)))

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def getDirectoryUuid(path: String): Option[String] = super.getDirectoryUuid(asCiPKType(getPkFromId(path)))

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def getDirectoryReference(path: String): Option[String] = super.getDirectoryReference(asCiPKType(getPkFromId(path)))
}

trait CiResolverQueries extends CiQueries {

  import CIS._

  lazy val UPDATE_SECURED_CI_FOR_PATH: String =
    sqlb"update $tableName set $secured_ci = ? where $path = ? or ($path like ? and $secured_ci = ?)"

  lazy val UPDATE_SECURED_CI_FOR_PATH_AND_NULL_SECURED_CI: String =
    sqlb"update $tableName set $secured_ci = ? where $path = ? or ($path like ? and $secured_ci is null)"

}
