package com.xebialabs.deployit.repository.sql.specific.tables

import com.xebialabs.deployit.core.sql._
import com.xebialabs.deployit.core.sql.batch.{BatchCommand, BatchCommandWithSetter}
import com.xebialabs.deployit.core.sql.spring.Setter
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.sql.base._
import com.xebialabs.deployit.repository.sql.reader.properties.{CiDataProvider, CiGroupCiDataProvider}
import com.xebialabs.deployit.repository.sql.specific.TypeSpecificPersisterFactory
import com.xebialabs.deployit.repository.sql.specific.configurable.ConfigurableTypeSpecificPersisterFactory.CiFromMapReader
import com.xebialabs.deployit.repository.sql.specific.configurable._
import org.springframework.jdbc.core.JdbcTemplate

import java.util.stream.Collectors
import java.util.{List => JList}
import scala.jdk.CollectionConverters._

abstract class ListPropertyTable[T, V](val ct: Class[T])
  extends MultiValuePropertyTable[JList[T], JList[V]] with Queries {

  val idxColumn: ColumnName
  val valueColumn: ColumnName

  override val selectDescriptor: PropertyTableSelectDescriptor = PropertyTableSelectDescriptor(
    table, List(valueColumn), idColumn, List(OrderBy.asc(idxColumn))
  )

  val INSERT = sqlb"insert into $table ($idColumn, $idxColumn, $valueColumn) values (?, ?, ?)"
  val UPDATE = sqlb"update $table set $valueColumn = ? where $idColumn = ? and $idxColumn = ?"
  val DELETE = sqlb"delete from $table where $idColumn = ? and $idxColumn = ?"

  def toParam(value: T): Any

  protected def ensureT(value: Any): T = value match {
    case matches: T => matches
    case _ => throw new IllegalArgumentException(s"Value $value doesn't match requested type ${ct.getCanonicalName}")
  }

  private def retrieveValueList[X](pk: CiPKType)(implicit ciDataProvider: CiDataProvider): JList[X] =
    ciDataProvider
      .getPropertyTableValues(pk, selectDescriptor)
      .asScala
      .map(map => ensureT(map.get(valueColumn.name)).asInstanceOf[X])
      .asJava

  override val reader: PropertyTableReader[JList[V]] = new PropertyTableReader[JList[V]] {
    override def readProperty(pk: CiPKType)
                             (implicit ciDataProvider: CiDataProvider,
                              readFromMap: CiFromMapReader,
                              typeSpecificPersisterFactories: JList[TypeSpecificPersisterFactory]): JList[V] =
      retrieveValueList(pk)
  }

  override val inserter: PropertyTableInserter[JList[T]] = new PropertyTableInserter[JList[T]] {
    override def insertTypedProperty(pk: CiPKType, value: JList[T], handler: ErrorHandler): Unit = {
      for (i <- 0 until value.size()) {
        handler {
          jdbcTemplate.update(INSERT, Setter(Seq(pk, i, toParam(value.get(i)))))
        }
      }
    }

    override def batchInsertTypedProperty(pk: CiPKType, values: JList[T]): List[BatchCommandWithSetter] = {
      values.asScala.toList.zipWithIndex.map {
        case (value, i) => BatchCommandWithSetter(INSERT, Setter(Seq(pk, i, toParam(value))))
      }
    }
  }

  override val updater: PropertyTableUpdater[JList[T]] = new PropertyTableUpdater[JList[T]] {
    private def zipAllOptionWithIndex[X](left: Vector[X], right: Vector[X]): LazyList[((Option[X], Option[X]), Int)] =
      LazyList.range(0, math.max(left.length, right.length)).map { idx => ((left.lift(idx), right.lift(idx)), idx) }

    override def updateTypedProperty(pk: CiPKType, newValues: JList[T])(implicit ciDataProvider: CiDataProvider): Unit =
      zipAllOptionWithIndex(newValues.asScala.toVector, retrieveValueList(pk).asScala.toVector).foreach {
        case ((Some(newValue), Some(oldValue)), idx) if oldValue != newValue =>
          jdbcTemplate.update(UPDATE, Setter(Seq(newValue, pk, idx)))
        case ((Some(newValue), None), idx) =>
          jdbcTemplate.update(INSERT, Setter(Seq(pk, idx, newValue)))
        case ((None, Some(_)), idx) =>
          jdbcTemplate.update(DELETE, Setter(Seq(pk, idx)))
        case _ =>
      }

    override def batchUpdateTypedProperty(pk: CiPKType, newValues: JList[T])
                                         (implicit ciDataProvider: CiDataProvider): List[BatchCommandWithSetter] =
      zipAllOptionWithIndex(newValues.asScala.toVector, retrieveValueList(pk).asScala.toVector)
        .foldLeft(Vector[BatchCommandWithSetter]()) {
          case (acc, ((Some(newValue), Some(oldValue)), idx)) if oldValue != newValue =>
            acc :+ BatchCommand(UPDATE, Setter(Seq(newValue, pk, idx)))
          case (acc, ((Some(newValue), None), idx)) =>
            acc :+ BatchCommand(INSERT, Setter(Seq(pk, idx, newValue)))
          case (acc, ((None, Some(_)), idx)) =>
            acc :+ BatchCommand(DELETE, Setter(Seq(pk, idx)))
          case (acc, _) => acc
        }
        .toList
  }
}

class ListCiPropertyTable(val table: TableName, val idColumn: ColumnName, val idxColumn: ColumnName, val valueColumn: ColumnName)
                         (implicit val jdbcTemplate: JdbcTemplate, implicit val schemaInfo: SchemaInfo)
  extends ListPropertyTable[CiPKType, ConfigurationItem](classOf[CiPKType])
    with CiCollectionPropertyTable[JList[CiPKType], JList[ConfigurationItem]] {
  override implicit val selectDescriptor: PropertyTableSelectDescriptor = PropertyTableSelectDescriptor(
    table, List(valueColumn), idColumn, List(OrderBy.asc(idxColumn)), ciRefColumn = Some(valueColumn)
  )

  override val reader: PropertyTableReader[JList[ConfigurationItem]] = new PropertyTableReader[JList[ConfigurationItem]] {
    override def readProperty(pk: CiPKType)(implicit ciDataProvider: CiDataProvider,
                                            readFromMap: CiFromMapReader,
                                            typeSpecificPersisterFactories: JList[TypeSpecificPersisterFactory]): JList[ConfigurationItem] = {
      val (lists, ciPkAndTypes) = retrieveMaps(pk)
      val setDataProvider = new CiGroupCiDataProvider(ciPkAndTypes)
      lists.stream().map[ConfigurationItem](readFromMap(_, setDataProvider)).collect(Collectors.toList())
    }
  }

  override def toParam(value: CiPKType): Any = asCiPKType(value)

  override protected def ensureT(value: Any): CiPKType = asCiPKType(value)
}

class ListStringPropertyTable(val table: TableName, val idColumn: ColumnName, val idxColumn: ColumnName, val valueColumn: ColumnName)
                             (implicit val jdbcTemplate: JdbcTemplate, implicit val schemaInfo: SchemaInfo)
  extends ListPropertyTable[String, String](classOf[String])
    with StringCollectionPropertyTable[JList[String], JList[String]] {
  override def toParam(value: String): Any = value
}
