package com.xebialabs.deployit.repository.sql.artifacts

import com.xebialabs.deployit.core.sql.spring.Setter
import com.xebialabs.deployit.core.sql.{Queries, SchemaInfo}
import com.xebialabs.deployit.repository.sql.artifacts.model.ArtifactFileInfo
import com.xebialabs.deployit.repository.sql.base.{CiPKType, idToPath}
import com.xebialabs.deployit.sql.base.schema.CIS
import com.xebialabs.deployit.repository.sql.persisters.SourceArtifactSchema
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.dao.support.DataAccessUtils.singleResult
import org.springframework.jdbc.core.{JdbcTemplate, SingleColumnRowMapper}
import org.springframework.stereotype.Component

import java.sql.ResultSet
import scala.jdk.CollectionConverters._

@Component
class SourceArtifactRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate)
                              (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends ArtifactRepository with SourceArtifactQueries {

  def updateFilename(pk: CiPKType, filename: String): Unit = {
    jdbcTemplate.update(UPDATE_ARTIFACT_FILENAME, filename, pk)
  }

  def getFilename(id: String): String = {
    singleResult(jdbcTemplate.query(SELECT_ARTIFACT_FILENAME, new SingleColumnRowMapper(classOf[String]), idToPath(id)))
  }

  def count(params: SourceArtifactQueryParams): Int = {
    val builder = SourceArtifactQueryBuilder.build(params.copy(count = true))
    jdbcTemplate
      .query(builder.query, Setter(builder.parameters), (rs: ResultSet, _: Int) => rs.getInt(1))
      .asScala
      .headOption.getOrElse(0)
  }

  def list(params: SourceArtifactQueryParams): List[ArtifactFileInfo] = {
    val builder = SourceArtifactQueryBuilder.build(params)
    jdbcTemplate.query(builder.query, Setter(builder.parameters), (rs: ResultSet, _: Int) =>
      ArtifactFileInfo(
        ciPath = rs.getString(1),
        fileName = rs.getString(2)
      )
    ).asScala.toList
  }
}

trait SourceArtifactQueries extends Queries {

  import SourceArtifactSchema._

  val SELECT_ARTIFACT_FILENAME =
    sqlb"""select $filename from $tableName sa
          |inner join ${CIS.tableName} ci
          |on (sa.$ID = ci.${CIS.ID})
          |where ci.${CIS.path} = ?"""

  val UPDATE_ARTIFACT_FILENAME = sqlb"update $tableName set $filename = ? where $ID = ?"
}
