package com.xebialabs.deployit.repository.sql.specific.tables

import com.xebialabs.deployit.core.sql.spring.{MapRowMapper, Setter}
import com.xebialabs.deployit.core.sql.util._
import com.xebialabs.deployit.core.sql.{ColumnName, Queries, SelectBuilder, TableName, SqlCondition => cond}
import com.xebialabs.deployit.repository.sql.base._
import com.xebialabs.deployit.repository.sql.specific.configurable.{PropertyTable, PropertyTableDeleter, PropertyTableReferenceFinder}
import org.springframework.jdbc.core.JdbcTemplate

import scala.jdk.CollectionConverters._

abstract class MultiValuePropertyTable[T, V]
  extends PropertyTable[T, V] with Queries {
  val table: TableName
  val idColumn: ColumnName
  val valueColumn: ColumnName
  val jdbcTemplate: JdbcTemplate
  val DELETE_ALL = sqlb"delete from $table where $idColumn = ?"
  override val deleter: PropertyTableDeleter = (pk: CiPKType) => jdbcTemplate.update(DELETE_ALL, pk)
}

trait CiCollectionPropertyTable[T, V] extends MultiValuePropertyTable[T, V] with Queries {
  val ct: Class[CiPKType]
  val valueColumn: ColumnName

  override val referenceFinder: PropertyTableReferenceFinder = new CiPropertyTableReferenceFinder

  class CiPropertyTableReferenceFinder extends PropertyTableReferenceFinder {
    override def findReferences(pks: Seq[CiPKType]): List[(CiPKType, CiPKType)] = {
      queryWithInClause(pks) { group =>
        findReferencesBuilder(group).map { builder =>
          builder.select(valueColumn)
          jdbcTemplate.query(builder.query, Setter(builder.parameters), MapRowMapper).asScala.map { map =>
            (asCiPKType(map.get(idColumn.name)), asCiPKType(map.get(valueColumn.name)))
          }
        }
      }.flatten
    }

    override def findReferencesBuilder(pks: Seq[CiPKType]): Option[SelectBuilder] = {
      val builder = new SelectBuilder(table)
        .distinct()
        .select(idColumn)
        .where(cond.in(valueColumn, pks))
      Option(builder)
    }
  }

}

trait StringCollectionPropertyTable[T, V] extends MultiValuePropertyTable[T, V] {
  override val referenceFinder: PropertyTableReferenceFinder = new StringPropertyTableReferenceFinder

  class StringPropertyTableReferenceFinder extends PropertyTableReferenceFinder {
    override def findReferences(pks: Seq[CiPKType]): Seq[(CiPKType, CiPKType)] = Nil

    override def findReferencesBuilder(pks: Seq[CiPKType]): Option[SelectBuilder] = None
  }

}
