package com.xebialabs.deployit.repository.sql

import com.xebialabs.deployit.checks.Checks
import com.xebialabs.deployit.core.sql.{ColumnName, ConditionOnColumn, SchemaInfo, SelectBuilder, SqlCondition => cond, SqlFunction => func}
import com.xebialabs.deployit.plugin.api.reflect.{DescriptorRegistry, Type}
import com.xebialabs.deployit.repository.sql.base.idToPath
import com.xebialabs.deployit.sql.base.schema.CIS
import org.joda.time.DateTime

import java.util
import scala.jdk.CollectionConverters._
import scala.util.Try

object CiConditions {

  import CIS._

  def idToPK(id: String)(implicit schemaInfo: SchemaInfo): SelectBuilder =
    new SelectBuilder(CIS.tableName).select(CIS.ID).where(cond.equals(CIS.path, idToPath(id)))

  def ofType(ciType: Type): cond =
    cond.in(ci_type, (ciType :: DescriptorRegistry.getSubtypes(ciType).asScala.toList).map(_.toString))

  def childOf(parent: String)(implicit schemaInfo: SchemaInfo): cond =
    if (parent equals "/")
      cond.equalsNull(parent_id)
    else
      cond.subselect(parent_id, new SelectBuilder(tableName).select(ID).where(cond.equals(path, idToPath(parent))))

  def descendantOf(ancestor: String): cond = cond.like(path, idToPath(ancestor) + "/%")

  def nameMatch(value: String, exact: Boolean): cond = new StringMatchCondition(name, value, exact)

  def internalIdMatch(internalId: Int): cond = cond.equals(ID, internalId)

  def idMatch(value: String, exact: Boolean): cond = new StringMatchCondition(path, value, exact)

  def modifiedBefore(time: DateTime): cond = cond.before(modified_at, time)

  def modifiedAfter(time: DateTime): cond = cond.after(modified_at, time)

  def hasProperties(properties: util.Map[String, String])(implicit schemaInfo: SchemaInfo): cond = {
    import com.xebialabs.deployit.sql.base.schema.CI_PROPERTIES
    cond.subselect(ID,
      properties.asScala.foldLeft(new SelectBuilder(CI_PROPERTIES.tableName).select(CI_PROPERTIES.ci_id)) { (selectBuilder: SelectBuilder, entry: (String, String)) =>
        validateEscapeSearchString(entry._2)
        selectBuilder.where(cond.and(Seq(
          cond.equals(CI_PROPERTIES.name, entry._1),
          Try(cond.equals(CI_PROPERTIES.boolean_value, entry._2.toBoolean)).getOrElse(cond.like(func.lower(CI_PROPERTIES.string_value), entry._2.toLowerCase))
        )))
      }
    )
  }

  private[sql] def validateEscapeSearchString(str: String): Unit = {
    def isBackslash(idx: Int) = str(idx) == '\\'

    def nextIsEscape(idx: Int) = idx < str.length - 1 && (str(idx + 1) == '_' || str(idx + 1) == '%')

    for (i <- 0 until str.length) {
      Checks.checkArgument(!(isBackslash(i) && !nextIsEscape(i)), "Invalid search string '%s'. Escape character '\\' " +
        "must be followed by another escape character, '_', or '%%'. It cannot be followed by any other character or be at the end of the pattern.", str)
    }
  }
}

class StringMatchCondition(override val column: ColumnName, value: String, exact: Boolean) extends ConditionOnColumn {
  private[this] def isWildCardBasedSearch(name: String): Boolean = {
    def isPercentage(idx: Int) = name(idx) == '%'

    def previousIsSlash(idx: Int) = idx > 0 && name(idx - 1) == '\\'

    for (i <- 0 until name.length) {
      if (isPercentage(i) && !previousIsSlash(i)) return true
    }
    false
  }

  private[this] val delegate =
    value match {
      case v if exact =>
        cond.equals(column, v)
      case v if isWildCardBasedSearch(v) =>
        CiConditions.validateEscapeSearchString(v)
        cond.likeEscaped(func.lower(column), v.toLowerCase())
      case v =>
        cond.equals(column, v.replace("\\%", "%"))
    }

  override def build(tableAlias: Option[String])(implicit schemaInfo: SchemaInfo): String = delegate.build(tableAlias)

  override def arguments: Seq[Any] = delegate.arguments
}
