package com.xebialabs.deployit.listener.placeholders

import com.xebialabs.deployit.engine.spi.event._
import com.xebialabs.deployit.engine.spi.event.DeployitEventListener
import com.xebialabs.deployit.repository.placeholders.PlaceholderRepositoryHolder
import grizzled.slf4j.Logging
import nl.javadude.t2bus.Subscribe

import scala.jdk.CollectionConverters._

@DeployitEventListener
class PlaceholderEventListener extends Logging {

  @Subscribe
  def onCreate(event: CisCreatedEvent): Unit = {
    val cis = event.getCis.asScala
    logger.trace(s"Inserting placeholders for $cis.")
    PlaceholderRepositoryHolder.getPlaceholderRepository.savePlaceholders(cis.toList)
  }

  @Subscribe
  def onCopy(event: CiCopiedEvent): Unit = {
    logger.trace(s"Copying placeholders from ${event.getCi.getId} to ${event.getNewId}.")
    PlaceholderRepositoryHolder.getPlaceholderRepository.copyPlaceholders(oldPath(event), newPath(event))
  }

  @Subscribe
  def onUpdate(event: CisUpdatedEvent): Unit = {
    val cis = event.getCis.asScala
    logger.trace(s"Updating placeholders for $cis.")
    PlaceholderRepositoryHolder.getPlaceholderRepository.updatePlaceholders(cis.toList)
  }

  @Subscribe
  def onDelete(event: CisDeletedEvent): Unit = {
    val cis = event.getCis.asScala
    logger.trace(s"Deleting placeholders for $cis.")
    PlaceholderRepositoryHolder.getPlaceholderRepository.deletePlaceholders(cis.toList)
  }

  @Subscribe
  def onRename(event: CiRenamedEvent): Unit = {
    handlePathUpdate(oldPath(event), newPath(event))
  }

  @Subscribe
  def onMove(event: CiMovedEvent): Unit = {
    handlePathUpdate(oldPath(event), newPath(event))
  }

  private def handlePathUpdate(oldPath: String, newPath: String): Unit = {
    logger.trace(s"Updating path for placeholders $oldPath -> $newPath.")
    PlaceholderRepositoryHolder.getPlaceholderRepository.updatePath(oldPath, newPath)
  }
}
