package com.xebialabs.deployit.repository.sql

import ai.digital.deploy.permissions.client.{ReferencedPermissionServiceClient, RolePrincipalsServiceClient}
import com.xebialabs.deployit.core.sql.spring.Setter
import com.xebialabs.deployit.core.sql.{OrderBy, SchemaInfo, SelectBuilder, SqlFunction, SqlCondition => cond}
import com.xebialabs.deployit.engine.api.dto
import com.xebialabs.deployit.engine.api.dto.{ConfigurationItemId, Paging}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.SatelliteRepository
import com.xebialabs.deployit.repository.sql.CiConditions._
import com.xebialabs.deployit.repository.sql.base._
import com.xebialabs.deployit.repository.sql.persisters.HostSchema
import com.xebialabs.deployit.security.PermissionChecker
import com.xebialabs.deployit.security.sql.SqlPermissionFilter
import com.xebialabs.deployit.sql.base.schema._
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository
import org.springframework.transaction.annotation.Transactional

import java.sql.ResultSet
import java.util
import scala.jdk.CollectionConverters._

@Repository
@Transactional("mainTransactionManager")
class SatelliteRepositoryImpl(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                              @Autowired override val referencedPermissionServiceClient: ReferencedPermissionServiceClient,
                              @Autowired override val rolePrincipalsServiceClient: RolePrincipalsServiceClient,
                              @Autowired override val checker: PermissionChecker,
                              @Autowired val ciRepository: CiRepository)
                             (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo) extends CiQueries with SqlPermissionFilter with SatelliteRepository {

  val LOCALHOST: Type = Type.valueOf("overthere.LocalHost")

  private def getAllHostsBuilder(satelliteId: String, hostNamePattern: Option[String]): SelectBuilder = {

    val satelliteSelector = idToPK(satelliteId)

    val localhostCondition = cond.and(Seq(
      // Type is LocalHost
      ofType(LOCALHOST),
      // Has "satellite" property that references the satellite.
      cond.subselect(CIS.ID, new SelectBuilder(CI_PROPERTIES.tableName).select(CI_PROPERTIES.ci_id)
        .where(cond.and(Seq(cond.equals(CI_PROPERTIES.name, "satellite"), cond.subselect(CI_PROPERTIES.ci_ref_value, satelliteSelector)))))
    ))

    val hostCondition = cond.subselect(CIS.ID, new SelectBuilder(HostSchema.tableName).select(HostSchema.ID).where(cond.subselect(HostSchema.satellite_id, satelliteSelector)))

    val builder = new SelectBuilder(CIS.tableName)
      .where(cond.or(Seq(localhostCondition, hostCondition)))

    hostNamePattern match {
      case Some(pattern) if pattern.nonEmpty =>
        builder.where(cond.like(SqlFunction.lower(CIS.name), s"%${pattern.toLowerCase}%"))
      case _ =>
    }

    addReadPermission(builder, CIS.secured_directory_ref)
    builder
  }

  override def countHosts(satelliteId: String, hostNamePattern: String = ""): Int = {
    val allHosts = getAllHostsBuilder(satelliteId, Option(hostNamePattern))

    allHosts.select(SqlFunction.countAll)
    jdbcTemplate.query(
      allHosts.query,
      Setter(allHosts.parameters),
      (rs: ResultSet, _: Int) => rs.getInt(1)
    ).asScala.headOption.getOrElse(0)
  }

  override def listHosts(satelliteId: String, hostNamePattern: String, paging: Paging, order: dto.Ordering): util.List[ConfigurationItemId] = {
    val allHosts = getAllHostsBuilder(satelliteId, Option(hostNamePattern))

    allHosts.select(CIS.path).select(CIS.ci_type)

    if (paging != null) {
      allHosts.showPage(paging.page, paging.resultsPerPage)
    }

    if (order != null) {
      allHosts.orderBy(if (order.isAscending) OrderBy.asc(CIS.path) else OrderBy.desc(CIS.path))
    }

    jdbcTemplate.query(allHosts.query, Setter(allHosts.parameters), new RowMapper[ConfigurationItemId] {
      override def mapRow(rs: ResultSet, rowNum: Int): ConfigurationItemId =
        new ConfigurationItemId(pathToId(rs.getString(1)), Type.valueOf(rs.getString(2)))
    })
  }
}
