package com.xebialabs.deployit.repository.sql

import com.xebialabs.deployit.core.sql.{ColumnName, Queries, SchemaInfo, TableName}
import com.xebialabs.deployit.repository.XlMetadataRepository
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.dao.{DuplicateKeyException, EmptyResultDataAccessException}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional

@Component
@Transactional("mainTransactionManager")
class SqlXlMetadataRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate)
                             (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends XlMetadataRepository with XlMetadataQueries {

  override def upsertValue(name: String, value: String): Unit = {
    upsertColumn(name, value, INSERT_VALUE, UPDATE_VALUE);
  }

  override def get(name: String): String = {
    try {
      jdbcTemplate.queryForObject(SELECT, classOf[String], name)
    } catch {
      case _: EmptyResultDataAccessException => null
      case e: Exception => throw new RuntimeException(s"Error while retrieving metadata with name $name", e)
    }
  }

  override def upsertContent(name: String, content: String): Unit = {
    upsertColumn(name, content, INSERT_CONTENT, UPDATE_CONTENT);
  }

  private def upsertColumn(name: String, value: String, insertQuery: String, updateQuery: String): Unit = {
    try {
      val rowsUpdated = jdbcTemplate.update(updateQuery, value, name)
      if (rowsUpdated == 0) {
        jdbcTemplate.update(insertQuery, name, value)
      }
    } catch {
      case _: DuplicateKeyException =>
      case e: Exception => throw new RuntimeException(s"Error updating metadata with name $name", e)
    }
  }

  override def getContent(name: String): String = {
    try {
      jdbcTemplate.queryForObject(SELECT_CONTENT, classOf[String], name)
    } catch {
      case _: EmptyResultDataAccessException => null
      case e: Exception => throw new RuntimeException(s"Error while retrieving metadata content with name $name", e)
    }
  }
}

  object XlMetadataSchema {
    val tableName: TableName = TableName("XL_METADATA")

    val name: ColumnName = ColumnName("name")
    val value: ColumnName = ColumnName("value")
    val content: ColumnName = ColumnName("content")
  }

  trait XlMetadataQueries extends Queries {

    import XlMetadataSchema._

    val SELECT = sqlb"select $value from $tableName where $name = ?"
    val INSERT_VALUE = sqlb"insert into $tableName ($name, $value) values (?, ?)"
    val UPDATE_VALUE = sqlb"update $tableName set $value = ? where $name = ?"
    val INSERT_CONTENT = sqlb"insert into $tableName ($name, $content) values (?, ?)"
    val UPDATE_CONTENT = sqlb"update $tableName set $content = ? where $name = ?"
    val SELECT_CONTENT = sqlb"select $content from $tableName where $name = ?"
  }
