package com.xebialabs.deployit.repository.sql

import ai.digital.deploy.permissions.client.{ReferencedPermissionServiceClient, RolePrincipalsServiceClient}
import com.xebialabs.deployit.core.sql.spring.Setter
import com.xebialabs.deployit.core.sql.{Queries, SelectBuilder, SqlCondition => cond, SqlFunction => func, _}
import com.xebialabs.deployit.engine.api.dto.{ConfigurationItemId, Ordering, Paging}
import com.xebialabs.deployit.plugin.api.reflect.{DescriptorRegistry, Type}
import com.xebialabs.deployit.repository.DeployedApplicationsRepository
import com.xebialabs.deployit.repository.sql.CiConditions._
import com.xebialabs.deployit.repository.sql.base.{idToPath, pathToId}
import com.xebialabs.deployit.security.PermissionChecker
import com.xebialabs.deployit.security.sql.{CiResolver, SqlPermissionFilter}
import com.xebialabs.deployit.sql.base.schema.CIS.{ci_type, name, path}
import com.xebialabs.deployit.sql.base.schema.{CIS, CI_PROPERTIES}
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository
import org.springframework.transaction.annotation.Transactional

import java.sql.ResultSet
import scala.jdk.CollectionConverters._

@Repository
@Transactional("mainTransactionManager")
class SqlDeployedApplicationsRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                        @Autowired override val checker: PermissionChecker,
                                        @Autowired override val referencedPermissionServiceClient: ReferencedPermissionServiceClient,
                                        @Autowired override val rolePrincipalsServiceClient: RolePrincipalsServiceClient,
                                        @Autowired val ciResolver: CiResolver)
                                       (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends DeployedApps with SqlPermissionFilter with DeployedApplicationsRepository {

  val DEPLOYED_APP_TYPE: Type = Type.valueOf("udm.DeployedApplication")
  val ENV_DEFAULT_ORDERING: Ordering = new Ordering("id:asc")
  val HOST_DEFAULT_ORDERING: Ordering = new Ordering("name:asc")
  val DEPLOYED_APP_DEFAULT_ORDERING: Ordering = new Ordering("id:asc")

  override def find(deployedAppName: String, path: String, exactPath: Boolean): List[ConfigurationItemId] = {
    val selectBuilder = getBaseApplicationStatusSelectBuilder(FIND_APPLICATION_STATUS, deployedAppName, path, exactPath)
    jdbcTemplate
      .query(selectBuilder.query, Setter(selectBuilder.parameters), mapRowToEnvAppCiId)
      .asScala
      .toList
  }

  override def count(deployedAppName: String, path: String, exactPath: Boolean): Int = {
    val selectBuilder = getBaseApplicationStatusSelectBuilder(COUNT_APPLICATION_STATUS, deployedAppName, path, exactPath)
    jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), mapCount).asScala.headOption.getOrElse(0)
  }

  override def findByEnvironment(envId: String, deployedAppName: String, paging: Paging, order: Ordering): List[ConfigurationItemId] = {
    val selectBuilder = new SelectBuilder(CIS.tableName)
      .select(CIS.path)
      .select(CIS.ci_type)
      .where(ofType(DEPLOYED_APP_TYPE))
      .where(childOf(envId))
      .where(cond.like(func.lower(CIS.name), s"%${if (deployedAppName == null) "" else deployedAppName.toLowerCase}%"))

    addPaging(selectBuilder, paging)

    addSorting(selectBuilder, SqlDeployedApplicationsFields.envFieldMapping, order, ENV_DEFAULT_ORDERING)

    jdbcTemplate
      .query(selectBuilder.query, Setter(selectBuilder.parameters), mapRowToEnvAppCiId)
      .asScala
      .toList
  }

  override def countByEnvironment(envId: String, deployedAppName: String): Int = {
    val selectBuilder = new SelectBuilder(CIS.tableName)
      .select(func.countAll)
      .where(ofType(DEPLOYED_APP_TYPE))
      .where(childOf(envId))
      .where(cond.like(CIS.name, s"%${if (deployedAppName == null) "" else deployedAppName}%"))

    jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), mapCount).asScala.headOption.getOrElse(0)
  }

  override def findByHost(hostId: String, deployedAppName: String, paging: Paging, order: Ordering): List[ConfigurationItemId] = {
    val selectBuilder = getBaseHostAppSelectBuilder(FIND_BY_HOST, hostId, deployedAppName)

    addSorting(selectBuilder, SqlDeployedApplicationsFields.hostFieldMapping, order, HOST_DEFAULT_ORDERING)
    addPaging(selectBuilder, paging)
    addReadPermission(selectBuilder, CIS.secured_directory_ref.tableAlias("da"))

    jdbcTemplate
      .query(selectBuilder.query, Setter(selectBuilder.parameters), mapRowToHostAppCiId)
      .asScala
      .toList
  }

  override def countByHost(hostId: String, deployedAppName: String): Int = {
    val selectBuilder = getBaseHostAppSelectBuilder(COUNT_BY_HOST, hostId, deployedAppName)
    addReadPermission(selectBuilder, CIS.secured_directory_ref.tableAlias("da"))
    jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), mapCount).asScala.headOption.getOrElse(0)
  }

  private def addPaging(selectBuilder: SelectBuilder, paging: Paging): SelectBuilder =
    if (paging != null) selectBuilder.showPage(paging.page, paging.resultsPerPage) else selectBuilder

  private def addSorting(selectBuilder: SelectBuilder, fieldMapping: Map[String, Selectable], ordering: Ordering, defaultOrdering: Ordering = ENV_DEFAULT_ORDERING): SelectBuilder = {
    val order = if (ordering == null) defaultOrdering else ordering
    val columnToSort = fieldMapping(order.field)
    selectBuilder.orderBy(
      if (order.isAscending) OrderBy.asc(columnToSort) else OrderBy.desc(columnToSort)
    )
  }

  private def getBaseHostAppSelectBuilder(query: String, hostId: String, namePattern: String): SelectBuilder =
    new SelectFragmentBuilder(query)
      .where(cond.in(
        ci_type.tableAlias("da"),
        (DEPLOYED_APP_TYPE :: DescriptorRegistry.getSubtypes(DEPLOYED_APP_TYPE).asScala.toList).map(_.toString)
      ))
      .where(cond.like(path.tableAlias("d"), idToPath(hostId) + "/%"))
      .where(cond.like(func.lower(name.tableAlias("da")), s"%${if (namePattern == null) "" else namePattern.toLowerCase}%"))

  private def getBaseApplicationStatusSelectBuilder(query: String, namePattern: String, path: String, exactPath: Boolean) = {
    val selectFragmentBuilder = new SelectFragmentBuilder(query)
      .where(cond.in(
        ci_type.tableAlias("xc1"),
        (DEPLOYED_APP_TYPE :: DescriptorRegistry.getSubtypes(DEPLOYED_APP_TYPE).asScala.toList).map(_.toString)
      ))
      .where(cond.like(func.lower(CIS.name.tableAlias("xc1")), s"%${if (namePattern == null) "" else namePattern.toLowerCase}%"))
    val selectBuilder = Option(path) match {
      case Some(p) => selectFragmentBuilder.where(cond.like(CIS.path.tableAlias("xc2"), s"${idToPath(p)}${if (!exactPath) "%" else ""}"))
      case None => selectFragmentBuilder
    }
    addReadPermission(selectBuilder, CIS.secured_directory_ref.tableAlias("xc1"))
    addReadPermission(selectBuilder, CIS.secured_directory_ref.tableAlias("xc2"))
    selectBuilder
  }


  private val mapCount: RowMapper[Int] = (rs: ResultSet, _) =>
    rs.getInt(1)

  private val mapRowToEnvAppCiId: RowMapper[ConfigurationItemId] = (rs: ResultSet, _) =>
    new ConfigurationItemId(pathToId(rs.getString(CIS.path.name)), Type.valueOf(rs.getString(CIS.ci_type.name)))

  private val mapRowToHostAppCiId: RowMapper[ConfigurationItemId] = (rs: ResultSet, _) =>
    new ConfigurationItemId(pathToId(rs.getString(DeployedAppAliases.DeployedApp.ciPath)), Type.valueOf(rs.getString(DeployedAppAliases.DeployedApp.ciType)))

}

object SqlDeployedApplicationsFields {
  val PATH = "id"
  val MODIFIED_AT = "modifiedAt"
  val NAME = "name"
  val ENVIRONMENT = "environment"
  val envFieldMapping: Map[String, ColumnName] = Map(
    PATH -> CIS.path,
    MODIFIED_AT -> CIS.modified_at
  )
  val hostFieldMapping: Map[String, Selectable] = Map(
    NAME -> CIS.name.tableAlias("da"),
    MODIFIED_AT -> CIS.modified_at.tableAlias("da"),
    ENVIRONMENT -> CIS.path.tableAlias("env")
  )
  val deployedAppFieldMapping: Map[String, Selectable] = Map(
    PATH -> CIS.path.tableAlias("xc1"),
    NAME -> CIS.name.tableAlias("xc1"),
    MODIFIED_AT -> CIS.modified_at.tableAlias("xc1"),
  )
}

object DeployedAppAliases {

  object DeployedApp {
    val ciType = "da_ci_type"
    val ciPath = "da_ci_path"
    val ciName = "da_ci_name"
    val ciModifiedAt = "da_ci_modified_at"
  }

  object Env {
    val ciPath = "env_ci_path"
  }

}

trait DeployedApps extends Queries {
  val APP_TYPE: Type = Type.valueOf("udm.Application")

  lazy val FIND_BY_HOST: String =
    sqlb"DISTINCT da.${CIS.ci_type} ${DeployedAppAliases.DeployedApp.ciType}, " +
      sqlb"da.${CIS.path} ${DeployedAppAliases.DeployedApp.ciPath}, " +
      sqlb"da.${CIS.name} ${DeployedAppAliases.DeployedApp.ciName}, " +
      sqlb"da.${CIS.modified_at} ${DeployedAppAliases.DeployedApp.ciModifiedAt}, " +
      sqlb"env.${CIS.path} ${DeployedAppAliases.Env.ciPath} " +
      sqlb"from ${CIS.tableName} d " +
      sqlb"INNER JOIN ${CI_PROPERTIES.tableName} p ON d.${CIS.ID} = p.${CI_PROPERTIES.ci_ref_value}" +
      sqlb"INNER JOIN ${CIS.tableName} da ON da.${CIS.ID} = p.${CI_PROPERTIES.ci_id}" +
      sqlb"INNER JOIN ${CIS.tableName} env ON da.${CIS.parent_id} = env.${CIS.ID}"

  lazy val COUNT_BY_HOST: String =
    sqlb"count(DISTINCT da.${CIS.ID}) from ${CIS.tableName} d " +
      sqlb"INNER JOIN ${CI_PROPERTIES.tableName} p ON d.${CIS.ID} = p.${CI_PROPERTIES.ci_ref_value}" +
      sqlb"INNER JOIN ${CIS.tableName} da ON da.${CIS.ID} = p.${CI_PROPERTIES.ci_id}"

  lazy val FIND_APPLICATION_STATUS: String =
    sqlb"xc1.${CIS.path}, xc1.${CIS.ci_type} from ${CIS.tableName} xc1 " +
      sqlb"INNER JOIN ${CIS.tableName} xc2 " +
      sqlb"ON xc2.${CIS.ci_type} = '${String.valueOf(APP_TYPE)}' " +
      sqlb"and xc2.${CIS.name} = xc1.${CIS.name}"

  lazy val COUNT_APPLICATION_STATUS: String =
    sqlb"count(DISTINCT xc1.${CIS.ID}) from ${CIS.tableName} xc1 " +
      sqlb"INNER JOIN ${CIS.tableName} xc2 " +
      sqlb"ON xc2.${CIS.ci_type} = '${String.valueOf(APP_TYPE)}' " +
      sqlb"and xc2.${CIS.name} = xc1.${CIS.name}"
}
