package com.xebialabs.deployit.repository.sql

import java.sql.ResultSet
import java.util
import com.xebialabs.deployit.core.sql.spring.Setter
import com.xebialabs.deployit.core.sql.{JoinBuilder, OrderBy, SchemaInfo, SelectBuilder, UnionAllBuilder, SqlCondition => cond}
import com.xebialabs.deployit.repository.{DictionaryRepository, StringValueConverter}
import com.xebialabs.deployit.repository.sql.persisters.{DictionaryApplicationsSchema, DictionaryContainersSchema, DictionaryEncryptedEntriesSchema, DictionaryEntriesSchema, EnvironmentDictionariesSchema}
import com.xebialabs.deployit.repository.sql.base.idToPath
import com.xebialabs.deployit.sql.base.schema.CIS
import com.xebialabs.deployit.util.PasswordEncrypter
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, SingleColumnRowMapper}
import org.springframework.stereotype.Repository
import org.springframework.transaction.annotation.Transactional

import scala.jdk.CollectionConverters._

@Repository
@Transactional("mainTransactionManager")
class SqlDictionaryRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                              @Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo,
                              @Autowired val passwordEncrypter: PasswordEncrypter) extends DictionaryRepository {

  private val cisAlias = "cis"
  private val dicApplAlias = "xda"
  private val dicContAlias = "xcd"
  private val envDic = "xed"
  private val dicEntry = "xde"

  private val appCond = cond.equals(DictionaryApplicationsSchema.dictionary_id.tableAlias(dicApplAlias), EnvironmentDictionariesSchema.dictionary_id.tableAlias(envDic))
  private val contCond = cond.equals(DictionaryContainersSchema.dictionary_id.tableAlias(dicContAlias), EnvironmentDictionariesSchema.dictionary_id.tableAlias(envDic))

  private def getDictionaryByEnvironment(environment: String): util.Map[String, String] = {
    val envOnlySelectBuilder = selectBuilderForEnvOnly(environment).orderBy(OrderBy.desc(EnvironmentDictionariesSchema.idx))
    jdbcTemplate.query(envOnlySelectBuilder.query, Setter(envOnlySelectBuilder.parameters), (rs: ResultSet) => resultSetToMapExtractor(rs))
  }

  private def getDictionaryByEnvironmentAndApplication(environment: String, application: String): util.Map[String, String] = {
    val envAndAppSelectBuilder = new UnionAllBuilder(
      selectBuilderForEnvAndApp(environment, application),
      selectBuilderForEnvOnly(environment)).orderBy(OrderBy.desc(EnvironmentDictionariesSchema.idx))
    jdbcTemplate.query(envAndAppSelectBuilder.query, Setter(envAndAppSelectBuilder.parameters), (rs: ResultSet) => resultSetToMapExtractor(rs))
  }

  private def getDictionaryByEnvironmentAndContainer(environment: String, container: String): util.Map[String, String] = {
    val envAndContSelectBuilder = new UnionAllBuilder(
      selectBuilderForEnvAndCont(environment, container),
      selectBuilderForEnvOnly(environment)).orderBy(OrderBy.desc(EnvironmentDictionariesSchema.idx))
    jdbcTemplate.query(envAndContSelectBuilder.query, Setter(envAndContSelectBuilder.parameters), (rs: ResultSet) => resultSetToMapExtractor(rs))
  }

  private def getDictionaryByEnvironmentApplicationAndContainer(environment: String, application: String, container: String): util.Map[String, String] = {
    val appContQueryBuilder = new UnionAllBuilder(
      selectBuilderForEnvAppCont(environment, container, application),
      selectBuilderForEnvAndCont(environment, container),
      selectBuilderForEnvAndApp(environment, application),
      selectBuilderForEnvOnly(environment)).orderBy(OrderBy.desc(EnvironmentDictionariesSchema.idx))
    jdbcTemplate.query(appContQueryBuilder.query, Setter(appContQueryBuilder.parameters), (rs: ResultSet) => resultSetToMapExtractor(rs))
  }

  private def dictionaryUnion(selectBuildEnv: SelectBuilder): UnionAllBuilder = {
    new UnionAllBuilder(
      new JoinBuilder(new SelectBuilder(DictionaryEntriesSchema.tableName).as(dicEntry)
        .select(DictionaryEntriesSchema.key)
        .select(DictionaryEntriesSchema.value))
        .join(selectBuildEnv, cond.equals(DictionaryEntriesSchema.dictionary_id.tableAlias(dicEntry), EnvironmentDictionariesSchema.dictionary_id.tableAlias(envDic))),
      new JoinBuilder(new SelectBuilder(DictionaryEncryptedEntriesSchema.tableName).as(dicEntry)
        .select(DictionaryEncryptedEntriesSchema.key)
        .select(DictionaryEncryptedEntriesSchema.value))
        .join(selectBuildEnv, cond.equals(DictionaryEncryptedEntriesSchema.dictionary_id.tableAlias(dicEntry), EnvironmentDictionariesSchema.dictionary_id.tableAlias(envDic))))
  }

  private def getEnvironmentSelectBuilder(environment: String, applicationCondition: cond, containerCondition: cond, dicEnvConditionFlag: Boolean = true): SelectBuilder = {
    val condition: cond = getConditionSeq(environment, applicationCondition, containerCondition, dicEnvConditionFlag)
    getDicEnvBaseQuery
      .where(condition)
  }

  private def getConditionSeq(environment: String, applicationCondition: cond, containerCondition: cond, dicEnvConditionFlag: Boolean): cond = {
    val envCondition = cond.subselect(EnvironmentDictionariesSchema.environment_id, getCisQueryByPath(environment))
    val condition: cond = cond.and(Seq(envCondition, applicationCondition, containerCondition))
    if (dicEnvConditionFlag) {
      val dicEnvCondition = cond.equals(DictionaryEntriesSchema.dictionary_id.tableAlias(dicEntry),
        EnvironmentDictionariesSchema.dictionary_id.tableAlias(envDic))
      cond.and(Seq(condition, dicEnvCondition))
    } else {
      condition
    }
  }

  private def getApplicationSelectBuilder(application: String): SelectBuilder = {
    getDicAppBaseQuery.where(
      cond.and(
        Seq(
          cond.subselect(DictionaryApplicationsSchema.application_id, getCisQueryByPath(application)),
          appCond
        )
      ))
  }

  private def getContainerSelectBuilder(container: String): SelectBuilder = {
    getDicContBaseQuery.where(
      cond.and(
        Seq(
          cond.subselect(DictionaryContainersSchema.container_id, getCisQueryByPath(container)),
          contCond
        )
      ))
  }

  private def getDicEnvBaseQuery: SelectBuilder = {
    new SelectBuilder(EnvironmentDictionariesSchema.tableName).as(envDic)
      .select(EnvironmentDictionariesSchema.dictionary_id)
      .select(EnvironmentDictionariesSchema.idx)
  }

  private def getDicAppBaseQuery: SelectBuilder = {
    new SelectBuilder(DictionaryApplicationsSchema.tableName).as(dicApplAlias)
      .select(DictionaryApplicationsSchema.dictionary_id)
  }

  private def getDicContBaseQuery: SelectBuilder = {
    new SelectBuilder(DictionaryContainersSchema.tableName).as(dicContAlias)
      .select(DictionaryContainersSchema.dictionary_id)
  }

  private def getCisQueryByPath(path: String): SelectBuilder = {
    new SelectBuilder(CIS.tableName).as(cisAlias)
      .select(CIS.ID)
      .where(cond.equals(CIS.path, idToPath(path)))
  }

  private def resultSetToMapExtractor(rs: ResultSet): util.Map[String, String] = {
    val map = new util.HashMap[String, String]()
    val geStringValueConverter: StringValueConverter = new StringValueConverter(passwordEncrypter)
    while (rs.next()) {
      val stringVal = geStringValueConverter.convert(rs.getString(DictionaryEntriesSchema.value.name))
      map.put(rs.getString(DictionaryEntriesSchema.key.name), stringVal.toPublicFacingValue)
    }
    map
  }

  private def selectBuilderForEnvOnly(environment: String): UnionAllBuilder = {
    val xldDicApplication = getDicAppBaseQuery.distinct()
    val xldDicContainer = getDicContBaseQuery.distinct()

    val xldEnvironment = getEnvironmentSelectBuilder(
      environment,
      cond.subselectNot(EnvironmentDictionariesSchema.dictionary_id, xldDicApplication),
      cond.subselectNot(EnvironmentDictionariesSchema.dictionary_id, xldDicContainer),
      dicEnvConditionFlag = false)

    dictionaryUnion(xldEnvironment)
  }

  private def selectBuilderForEnvAndApp(environment: String, application: String): UnionAllBuilder = {
    val xldDicApplication = getApplicationSelectBuilder(application)
    val xldDicContainer = getDicContBaseQuery.distinct().where(contCond)

    val xldEnvironment = getEnvironmentSelectBuilder(
      environment,
      cond.subselect(EnvironmentDictionariesSchema.dictionary_id, xldDicApplication),
      cond.subselectNot(EnvironmentDictionariesSchema.dictionary_id, xldDicContainer))

    dictionaryUnion(xldEnvironment)
  }

  private def selectBuilderForEnvAndCont(environment: String, container: String): UnionAllBuilder = {
    val xldDicContainer = getContainerSelectBuilder(container)
    val xldDicApplication = getDicAppBaseQuery.distinct().where(appCond)

    val xldEnvironment = getEnvironmentSelectBuilder(
      environment,
      cond.subselectNot(EnvironmentDictionariesSchema.dictionary_id, xldDicApplication),
      cond.subselect(EnvironmentDictionariesSchema.dictionary_id, xldDicContainer))

    dictionaryUnion(xldEnvironment)
  }

  private def selectBuilderForEnvAppCont(environment: String, container: String, application: String): UnionAllBuilder = {
    val xldDicContainer = getContainerSelectBuilder(container)
    val xldDicApplication = getApplicationSelectBuilder(application)

    val xldEnvironment = getEnvironmentSelectBuilder(
      environment,
      cond.subselect(EnvironmentDictionariesSchema.dictionary_id, xldDicApplication),
      cond.subselect(EnvironmentDictionariesSchema.dictionary_id, xldDicContainer))

    dictionaryUnion(xldEnvironment)
  }

  private def getSelectedApplication(application: Option[String], applicationVersion: Option[String]): Option[String] = {
    applicationVersion match {
      case Some(version) =>
        val parentIdQuery = new SelectBuilder(CIS.tableName).as(cisAlias).select(CIS.parent_id).where(cond.equals(CIS.path, idToPath(version)))
        val appPathQuery = new SelectBuilder(CIS.tableName).as(cisAlias).select(CIS.path).where(cond.subselect(CIS.ID, parentIdQuery))
        jdbcTemplate.query(appPathQuery.query, Setter(appPathQuery.parameters), new SingleColumnRowMapper[String]()).asScala.headOption
      case None => application
    }
  }

  private def getDictionaryEntries(environment: String, selectedContainer: Option[String], selectedApplication: Option[String]): util.Map[String, String] = {
    (selectedApplication, selectedContainer) match {
      case (None, None) => getDictionaryByEnvironment(environment)
      case (Some(application), None) => getDictionaryByEnvironmentAndApplication(environment, application)
      case (None, Some(container)) => getDictionaryByEnvironmentAndContainer(environment, container)
      case (Some(application), Some(container)) => getDictionaryByEnvironmentApplicationAndContainer(environment, application, container)
    }
  }

  override def getDictionariesForEnv(environment: String, application: String, container: String, applicationVersion: String): util.Map[String, String] = {
    val selectedApplication: Option[String] = getSelectedApplication(Option(application), Option(applicationVersion))
    getDictionaryEntries(environment, Option(container), selectedApplication)
  }

}



