package com.xebialabs.deployit.repository.sql.placeholders

import java.lang.{Boolean => JBoolean}
import java.sql.{PreparedStatement, ResultSet}
import java.util.UUID

import ai.digital.configuration.central.deploy.ServerSideProperties
import com.google.common.base.Strings.nullToEmpty
import com.xebialabs.deployit.core.sql.spring.Setter
import com.xebialabs.deployit.core.sql.spring.Setter.setString
import com.xebialabs.deployit.core.sql.{SqlCondition => cond, _}
import com.xebialabs.deployit.engine.api.dto.{Ordering, Paging}
import com.xebialabs.deployit.plugin.api.deployment.ResolvedPlaceholder
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.placeholders.{ResolvedPlaceholderEntry, ResolvedPlaceholderRepository}
import com.xebialabs.deployit.security.sql.CiResolver
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{BatchPreparedStatementSetter, JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository
import org.springframework.transaction.annotation.Transactional

import scala.jdk.CollectionConverters._
import scala.language.implicitConversions

@Repository
@Transactional("mainTransactionManager")
class SqlResolvedPlaceholderRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                       @Autowired val ciResolver: CiResolver,
                                       @Autowired val serverSideProperties: ServerSideProperties)
                                      (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends ResolvedPlaceholderRepository with PlaceholdersQueries with ArchivedPlaceholdersQueries with Logging {

  val DEFAULT_ORDERING: Ordering = new Ordering("deployedAppId:asc")

  override def isPlaceholderEncrypted(key: String, containerPath: String, versionPath: String): Boolean = {
    jdbcTemplate.queryForList(IS_ENCRYPTED, classOf[JBoolean], key, containerPath, versionPath)
      .asScala.headOption match {
      case Some(isEncrypted) => isEncrypted
      case None => false
    }
  }

  override def saveResolvedPlaceholders(taskId: String, resolvedPlaceholders: java.util.Set[ResolvedPlaceholder]): Unit = {
    if (!resolvedPlaceholders.isEmpty) {
      if (taskId == null) {
        logger.error(s"Could not save resolved placeholders $resolvedPlaceholders. Reason: taskId == null.")
      } else {
        val placeholders = resolvedPlaceholders.asScala.toList

        jdbcTemplate.batchUpdate(INSERT, new BatchPreparedStatementSetter {
          override def getBatchSize: Int = placeholders.size

          override def setValues(ps: PreparedStatement, i: Int): Unit = {
            val placeholder = placeholders(i)
            ps.setString(1, UUID.randomUUID().toString)
            ps.setString(2, taskId)
            ps.setString(3, placeholder.key)
            setString(ps, 4, Option(placeholder.value).map(_.take(255)).orNull)
            setString(ps, 5, placeholder.value)
            ps.setBoolean(6, placeholder.isEncrypted)
            val containerId = nullToEmpty(placeholder.containerId)
            ps.setString(7, containerId)
            ps.setString(8, containerId.split("/").lastOption.getOrElse(""))
            ps.setString(9, placeholder.deployedAppId)
            ps.setString(10, placeholder.deployedAppId.split("/").lastOption.getOrElse(""))
            ps.setString(11, placeholder.dictionaryId)
            ps.setString(12, placeholder.dictionaryId.split("/").lastOption.getOrElse(""))
            ps.setString(13, placeholder.environmentId)
            ps.setString(14, placeholder.environmentId.split("/").lastOption.getOrElse(""))
            ps.setString(15, placeholder.versionId)
            ps.setInt(16, 0)
            ps.setInt(17, 0)
            ps.setInt(18, 0)
          }
        })
      }
    }
  }

  def deleteResolvedPlaceholders(deployedAppIds: List[String]): Unit = {
    jdbcTemplate.batchUpdate(DELETE, new BatchPreparedStatementSetter {
      override def getBatchSize: Int = deployedAppIds.size

      override def setValues(ps: PreparedStatement, i: Int): Unit = {
        val deployedAppId = deployedAppIds(i)
        ps.setString(1, deployedAppId)
      }
    })
  }

  override def getResolvedPlaceholder(deployedAppId: String): List[(String, ResolvedPlaceholderEntry)] = {
    val selectBuilder = new SelectBuilder(PlaceholdersSchema.tableName).
      select(PlaceholdersSchema.taskId).
      select(PlaceholdersSchema.isEncrypted).
      select(PlaceholdersSchema.key).
      select(PlaceholdersSchema.value).
      select(PlaceholdersSchema.fullValue).
      select(PlaceholdersSchema.containerId).
      select(PlaceholdersSchema.deployedAppId).
      select(PlaceholdersSchema.dictionaryId).
      select(PlaceholdersSchema.environmentId).
      select(PlaceholdersSchema.versionId).
      select(PlaceholdersSchema.containerDeleted).
      select(PlaceholdersSchema.dictionaryDeleted).
      select(PlaceholdersSchema.environmentDeleted).
      where(cond.equals(PlaceholdersSchema.deployedAppId, deployedAppId))

    jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), new RowMapper[(String, ResolvedPlaceholderEntry)] {
      implicit def int2bool(i: Int): Boolean = i == 1

      override def mapRow(rs: ResultSet, rowNum: Int): (String, ResolvedPlaceholderEntry) =
        (
          rs.getString(1),
          ResolvedPlaceholderEntry(
            rs.getBoolean(2),
            rs.getString(3),
            rs.getString(4),
            rs.getString(5),
            rs.getString(6),
            rs.getString(7),
            rs.getString(8),
            rs.getString(9),
            rs.getString(10),
            rs.getInt(11),
            rs.getInt(12),
            rs.getInt(13)
          )
        )
    }).asScala.toList
  }

  override def archiveResolvedPlaceholder(deployedAppId: String): Unit = {
    val placeholders = getResolvedPlaceholder(deployedAppId)
    if (serverSideProperties.placeholders.archive) {
      saveArchivedPlaceholders(placeholders)
    }
    deleteResolvedPlaceholders(List(deployedAppId))
  }

  private def saveArchivedPlaceholders(entries: List[(String, ResolvedPlaceholderEntry)]) = {
    if (entries.nonEmpty) {

      jdbcTemplate.batchUpdate(INSERT_ARCHIVE, new BatchPreparedStatementSetter {
        override def getBatchSize: Int = entries.size

        override def setValues(ps: PreparedStatement, i: Int): Unit = {
          implicit def bool2int(b: Boolean): Int = if (b) 1 else 0

          val entry: (String, ResolvedPlaceholderEntry) = entries(i)
          val taskId = entry._1
          val placeholderEntry = entry._2
          ps.setString(1, UUID.randomUUID().toString)
          ps.setString(2, taskId)
          ps.setString(3, placeholderEntry.key)
          setString(ps, 4, Option(placeholderEntry.value).map(_.take(255)).orNull)
          setString(ps, 5, placeholderEntry.fullValue)
          ps.setBoolean(6, placeholderEntry.isEncrypted)
          val containerId = nullToEmpty(placeholderEntry.containerId)
          ps.setString(7, containerId)
          ps.setString(8, containerId.split("/").lastOption.getOrElse(""))
          ps.setString(9, placeholderEntry.deployedAppId)
          ps.setString(10, placeholderEntry.deployedAppId.split("/").lastOption.getOrElse(""))
          ps.setString(11, placeholderEntry.dictionaryId)
          ps.setString(12, placeholderEntry.dictionaryId.split("/").lastOption.getOrElse(""))
          ps.setString(13, placeholderEntry.environmentId)
          ps.setString(14, placeholderEntry.environmentId.split("/").lastOption.getOrElse(""))
          ps.setString(15, placeholderEntry.versionId)
          ps.setInt(16, placeholderEntry.containerDeleted)
          ps.setInt(17, placeholderEntry.dictionaryDeleted)
          ps.setInt(18, placeholderEntry.environmentDeleted)
        }
      })
    }
  }

  private def countAllResolvedPlaceholders(subCondition: cond, schema: CommonPlaceholdersSchema): Int = {
    val builder = new SelectBuilder(schema.tableName).select(SqlFunction.countAll)
      .where(subCondition)

    jdbcTemplate.query(builder.query, Setter(builder.parameters), (rs: ResultSet, _: Int) =>
      rs.getInt(1)).asScala.headOption.getOrElse(0)
  }

  private def countResolvedPlaceholders(conditions: Option[List[cond]], subCondition: cond, schema: CommonPlaceholdersSchema): Int = {
    conditions match {
      case Some(sqlConditions) =>
        countAllResolvedPlaceholders(cond.and(sqlConditions.::(subCondition)), schema)
      case _ =>
        countAllResolvedPlaceholders(subCondition, schema)
    }
  }

  override def countAllResolvedPlaceholdersForEnvironment(environmentId: String, key: Option[String], value: Option[String], dictionary: Option[String],
                                                          deployedApplication: Option[String], host: Option[String]): Int = {
    val subCondition = cond.equals(PlaceholdersSchema.environmentId, environmentId)
    countResolvedPlaceholders(EnvironmentPlaceholderSearchParams.createConditions(key, value, dictionary, deployedApplication, host), subCondition, PlaceholdersSchema)
  }

  override def countAllResolvedPlaceholdersForContainer(containerId: String, key: Option[String], value: Option[String], dictionary: Option[String],
                                                        deployedApplication: Option[String], environment: Option[String]): Int = {
    val subCondition = cond.equals(PlaceholdersSchema.containerId, containerId)
    countResolvedPlaceholders(InfrastructurePlaceholderSearchParams.createConditions(key, value, dictionary, deployedApplication, environment), subCondition, PlaceholdersSchema)
  }

  override def markPlaceholderReferenceAsDeleted(ci: ConfigurationItem): Unit = {
    ci match {
      case deployedApplicationType() =>
        deleteResolvedPlaceholders(List(ci.getId))
      case baseContainerType() =>
        jdbcTemplate.update(MARK_CONTAINER_AS_DELETED, ci.getId)
      case dictionaryType() =>
        jdbcTemplate.update(MARK_DICTIONARY_AS_DELETED, ci.getId)
      case environmentType() =>
        jdbcTemplate.update(MARK_ENVIRONMENT_AS_DELETED, ci.getId)
      case _ =>
    }
  }

  override def markArchivedPlaceholderReferenceAsDeleted(ci: ConfigurationItem, placeholders: List[(String, ResolvedPlaceholderEntry)]): Unit = {
    if (serverSideProperties.placeholders.archive) {
      ci match {
        case deployedApplicationType() =>
          saveArchivedPlaceholders(placeholders)
        case baseContainerType() =>
          jdbcTemplate.update(MARK_CONTAINER_IN_ARCHIVE_AS_DELETED, ci.getId)
        case dictionaryType() =>
          jdbcTemplate.update(MARK_DICTIONARY_IN_ARCHIVE_AS_DELETED, ci.getId)
        case environmentType() =>
          jdbcTemplate.update(MARK_ENVIRONMENT_IN_ARCHIVE_AS_DELETED, ci.getId)
        case _ =>
      }
    }
  }

  override def updatePlaceholderReferencePaths(oldPath: String, newPath: String): Unit = {
    def pathToId(path: String): String = path.substring(1)

    val oldId = pathToId(oldPath)
    val newId = pathToId(newPath)
    val newName = newId.split("/").lastOption.getOrElse("")

    jdbcTemplate.update(UPDATE_CONTAINER_ID, newId, newName, oldId)
    jdbcTemplate.update(UPDATE_CONTAINER_CHILDREN_ID, newId + "/", (oldId.length + 2).asInstanceOf[Number], oldId + "/%")

    jdbcTemplate.update(UPDATE_DEPLOYED_APP_ID, newId, newName, oldId)
    jdbcTemplate.update(UPDATE_DEPLOYED_APP_CHILDREN_ID, newId + "/", (oldId.length + 2).asInstanceOf[Number], oldId + "/%")

    jdbcTemplate.update(UPDATE_DICTIONARY_ID, newId, newName, oldId)
    jdbcTemplate.update(UPDATE_DICTIONARY_CHILDREN_ID, newId + "/", (oldId.length + 2).asInstanceOf[Number], oldId + "/%")

    jdbcTemplate.update(UPDATE_ENVIRONMENT_ID, newId, newName, oldId)
    jdbcTemplate.update(UPDATE_ENVIRONMENT_CHILDREN_ID, newId + "/", (oldId.length + 2).asInstanceOf[Number], oldId + "/%")

    jdbcTemplate.update(UPDATE_VERSION_ID, newId, oldId)
    jdbcTemplate.update(UPDATE_VERSION_CHILDREN_ID, newId + "/", (oldId.length + 2).asInstanceOf[Number], oldId + "/%")
  }

  private def getResolvedPlaceholders(subCondition: cond, paging: Paging, order: Ordering, schema: CommonPlaceholdersSchema): List[ResolvedPlaceholder] = {
    val selectBuilder = new SelectBuilder(schema.tableName).
      select(schema.isEncrypted).
      select(schema.key).
      select(schema.value).
      select(schema.fullValue).
      select(schema.containerId).
      select(schema.containerDeleted).
      select(schema.deployedAppId).
      select(schema.dictionaryId).
      select(schema.dictionaryDeleted).
      select(schema.environmentId).
      select(schema.environmentDeleted).
      select(schema.versionId).
      where(subCondition)

    val fields = SqlPlaceholdersSelectQuery.orderFieldMapping.keySet.asJava

    val evaluatedOrder = if (order == null) DEFAULT_ORDERING else order

    evaluatedOrder.validate(fields)
    selectBuilder.orderBy(
      if (evaluatedOrder.isAscending) OrderBy.asc(SqlPlaceholdersSelectQuery.orderFieldMapping(evaluatedOrder.field))
      else OrderBy.desc(SqlPlaceholdersSelectQuery.orderFieldMapping(evaluatedOrder.field)))

    if (paging != null) {
      selectBuilder.showPage(paging.page, paging.resultsPerPage)
    }


    jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), new RowMapper[ResolvedPlaceholder] {
      override def mapRow(rs: ResultSet, rowNum: Int): ResolvedPlaceholder = {
        val value = rs.getString(3)
        val fullValue = rs.getString(4)
        ResolvedPlaceholder(
          rs.getBoolean(1),
          rs.getString(2),
          Option(fullValue).getOrElse(value),
          rs.getString(5),
          rs.getBoolean(6),
          rs.getString(7),
          rs.getString(8),
          rs.getBoolean(9),
          rs.getString(10),
          rs.getBoolean(11),
          rs.getString(12)
        )()
      }
    }).asScala.toList
  }

  private def getResolvedPlaceholders(conditions: Option[List[cond]], subCondition: cond, paging: Paging, order: Ordering,
                                      schema: CommonPlaceholdersSchema): List[ResolvedPlaceholder] = {
    conditions match {
      case Some(sqlConditions) =>
        getResolvedPlaceholders(cond.and(sqlConditions.::(subCondition)), paging, order, schema)
      case _ =>
        getResolvedPlaceholders(subCondition, paging, order, schema)
    }
  }

  override def getResolvedPlaceholdersForEnvironment(environmentId: String, key: Option[String], value: Option[String], dictionary: Option[String],
                                                     deployedApplication: Option[String], host: Option[String],
                                                     paging: Paging, order: Ordering): List[ResolvedPlaceholder] = {
    val subCondition = cond.equals(PlaceholdersSchema.environmentId, environmentId)
    getResolvedPlaceholders(EnvironmentPlaceholderSearchParams.createConditions(key, value, dictionary, deployedApplication, host), subCondition, paging, order, PlaceholdersSchema)
  }

  override def getResolvedPlaceholdersForContainer(containerId: String, key: Option[String], value: Option[String],
                                                   dictionary: Option[String], deployedApplication: Option[String],
                                                   environment: Option[String], paging: Paging, order: Ordering): List[ResolvedPlaceholder] = {
    val subCondition = cond.equals(PlaceholdersSchema.containerId, containerId)
    getResolvedPlaceholders(InfrastructurePlaceholderSearchParams.createConditions(key, value, dictionary, deployedApplication, environment), subCondition, paging, order, PlaceholdersSchema)
  }

  override def countAllArchivedResolvedPlaceholdersForEnvironment(environmentId: String, key: Option[String], value: Option[String], dictionary: Option[String],
                                                                  deployedApplication: Option[String], host: Option[String], taskId: Option[String]): Int = {
    val subCondition: cond = getSubConditionForEnvironmentIdAndTaskId(environmentId, taskId)
    countResolvedPlaceholders(EnvironmentPlaceholderSearchParams.createConditions(key, value, dictionary, deployedApplication, host), subCondition, ArchivedPlaceholdersSchema)
  }

  override def getArchivedResolvedPlaceholdersForEnvironment(environmentId: String, key: Option[String], value: Option[String], dictionary: Option[String],
                                                             deployedApplication: Option[String], host: Option[String], taskId: Option[String],
                                                             paging: Paging, order: Ordering): List[ResolvedPlaceholder] = {
    val subCondition: cond = getSubConditionForEnvironmentIdAndTaskId(environmentId, taskId)
    getResolvedPlaceholders(EnvironmentPlaceholderSearchParams.createConditions(key, value, dictionary, deployedApplication, host), subCondition, paging, order, ArchivedPlaceholdersSchema)
  }

  private def getSubConditionForEnvironmentIdAndTaskId(environmentId: String, taskId: Option[String]): cond = {
    val equalsEnvironmentIdCond = cond.equals(PlaceholdersSchema.environmentId, environmentId)
    taskId match {
      case Some(tid) => cond.and(Seq(
        equalsEnvironmentIdCond,
        cond.equals(PlaceholdersSchema.taskId, tid)
      ))
      case None => equalsEnvironmentIdCond
    }
  }
}

object SqlPlaceholdersSelectQuery {
  val orderFieldMapping: Map[String, ColumnName] = Map(
    "key" -> PlaceholdersSchema.key,
    "value" -> PlaceholdersSchema.value,
    "fullValue" -> PlaceholdersSchema.fullValue,
    "containerId" -> PlaceholdersSchema.containerId,
    "deployedAppId" -> PlaceholdersSchema.deployedAppId,
    "dictionaryId" -> PlaceholdersSchema.dictionaryId,
    "environmentId" -> PlaceholdersSchema.environmentId,
    "versionId" -> PlaceholdersSchema.versionId,
  )
}

trait CommonPlaceholdersSchema {
  val tableName: TableName
  var id: ColumnName = ColumnName("ID")
  val isEncrypted: ColumnName = ColumnName("encrypted")
  val key: ColumnName = ColumnName("key")
  val value: ColumnName = ColumnName("value")
  val fullValue: ColumnName = ColumnName("full_value")
  val containerId: ColumnName = ColumnName("container_id")
  val containerName: ColumnName = ColumnName("container_name")
  val containerDeleted: ColumnName = ColumnName("container_deleted")
  val deployedAppId: ColumnName = ColumnName("deployed_app_id")
  val deployedAppName: ColumnName = ColumnName("deployed_app_name")
  val dictionaryId: ColumnName = ColumnName("dictionary_id")
  val dictionaryName: ColumnName = ColumnName("dictionary_name")
  val dictionaryDeleted: ColumnName = ColumnName("dictionary_deleted")
  val environmentId: ColumnName = ColumnName("environment_id")
  val environmentName: ColumnName = ColumnName("environment_name")
  val environmentDeleted: ColumnName = ColumnName("environment_deleted")
  val versionId: ColumnName = ColumnName("version_id")
  val taskId: ColumnName = ColumnName("task_id")
}

object ArchivedPlaceholdersSchema extends CommonPlaceholdersSchema {
  val tableName: TableName = TableName("XLD_ARCHIVED_PLACEHOLDERS")
}

object PlaceholdersSchema extends CommonPlaceholdersSchema {
  val tableName: TableName = TableName("XLD_PLACEHOLDERS")
}

trait ArchivedPlaceholdersQueries extends Queries {

  import ArchivedPlaceholdersSchema._

  lazy val INSERT_ARCHIVE: String =
    sqlb"""insert into $tableName ($id, $taskId, $key, $value, $fullValue, $isEncrypted, $containerId, $containerName, $deployedAppId,
          | $deployedAppName, $dictionaryId, $dictionaryName, $environmentId, $environmentName, $versionId,
          | $containerDeleted, $dictionaryDeleted, $environmentDeleted)
          | values (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"""

  lazy val MARK_CONTAINER_IN_ARCHIVE_AS_DELETED: String = sqlb"update $tableName set $containerDeleted = 1 where $containerId = ?"

  lazy val MARK_ENVIRONMENT_IN_ARCHIVE_AS_DELETED: String = sqlb"update $tableName set $environmentDeleted = 1 where $environmentId = ?"

  lazy val MARK_DICTIONARY_IN_ARCHIVE_AS_DELETED: String = sqlb"update $tableName set $dictionaryDeleted = 1 where $dictionaryId = ?"
}

trait PlaceholdersQueries extends Queries {

  import PlaceholdersSchema._

  lazy val IS_ENCRYPTED: String = sqlb"select $isEncrypted from $tableName where $key = ? and $containerId = ? and $versionId = ?"

  lazy val DELETE: String = sqlb"delete from $tableName where $deployedAppId = ?"

  lazy val MARK_CONTAINER_AS_DELETED: String = sqlb"update $tableName set $containerDeleted = 1 where $containerId = ?"

  lazy val MARK_ENVIRONMENT_AS_DELETED: String = sqlb"update $tableName set $environmentDeleted = 1 where $environmentId = ?"

  lazy val MARK_DICTIONARY_AS_DELETED: String = sqlb"update $tableName set $dictionaryDeleted = 1 where $dictionaryId = ?"

  lazy val UPDATE_CONTAINER_ID: String = sqlb"update $tableName set $containerId = ?, $containerName = ? where $containerId = ? and $containerDeleted = 0"

  lazy val UPDATE_CONTAINER_CHILDREN_ID: String = {
    sqlb"update $tableName set $containerId = ${schemaInfo.sqlDialect.concat(schemaInfo.sqlDialect.paramWithCollation(), schemaInfo.sqlDialect.substr(containerId))} where $containerId like ? and $containerDeleted = 0"
  }

  lazy val UPDATE_DEPLOYED_APP_ID: String = sqlb"update $tableName set $deployedAppId = ?, $deployedAppName = ? where $deployedAppId = ?"

  lazy val UPDATE_DEPLOYED_APP_CHILDREN_ID: String = {
    sqlb"update $tableName set $deployedAppId = ${schemaInfo.sqlDialect.concat(schemaInfo.sqlDialect.paramWithCollation(), schemaInfo.sqlDialect.substr(deployedAppId))} where $deployedAppId like ?"
  }

  lazy val UPDATE_DICTIONARY_ID: String = sqlb"update $tableName set $dictionaryId = ?, $dictionaryName = ? where $dictionaryId = ? and $dictionaryDeleted = 0"

  lazy val UPDATE_DICTIONARY_CHILDREN_ID: String = {
    sqlb"update $tableName set $dictionaryId = ${schemaInfo.sqlDialect.concat(schemaInfo.sqlDialect.paramWithCollation(), schemaInfo.sqlDialect.substr(dictionaryId))} where $dictionaryId like ? and $dictionaryDeleted = 0"
  }

  lazy val UPDATE_ENVIRONMENT_ID: String = sqlb"update $tableName set $environmentId = ?, $environmentName = ? where $environmentId = ? and $environmentDeleted = 0"

  lazy val UPDATE_ENVIRONMENT_CHILDREN_ID: String = {
    sqlb"update $tableName set $environmentId = ${schemaInfo.sqlDialect.concat(schemaInfo.sqlDialect.paramWithCollation(), schemaInfo.sqlDialect.substr(environmentId))} where $environmentId like ? and $environmentDeleted = 0"
  }

  lazy val UPDATE_VERSION_ID: String = sqlb"update $tableName set $versionId = ? where $versionId = ?"

  lazy val UPDATE_VERSION_CHILDREN_ID: String = {
    sqlb"update $tableName set $versionId = ${schemaInfo.sqlDialect.concat(schemaInfo.sqlDialect.paramWithCollation(), schemaInfo.sqlDialect.substr(versionId))} where $versionId like ?"
  }

  lazy val INSERT: String =
    sqlb"""insert into $tableName ($id, $taskId, $key, $value, $fullValue, $isEncrypted, $containerId, $containerName,
          | $deployedAppId, $deployedAppName, $dictionaryId, $dictionaryName, $environmentId, $environmentName, $versionId,
          | $containerDeleted, $dictionaryDeleted, $environmentDeleted)
          | values (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"""
}

object deployedApplicationType {
  def unapply(ci: ConfigurationItem): Boolean = ci.getType.instanceOf(Type.valueOf("udm.DeployedApplication"))
}

object baseContainerType {
  def unapply(ci: ConfigurationItem): Boolean = ci.getType.instanceOf(Type.valueOf("udm.BaseContainer"))
}

object dictionaryType {
  def unapply(ci: ConfigurationItem): Boolean = ci.getType.instanceOf(Type.valueOf("udm.Dictionary"))
}

object environmentType {
  def unapply(ci: ConfigurationItem): Boolean = ci.getType.instanceOf(Type.valueOf("udm.Environment"))
}
