package com.xebialabs.deployit.repository.sql.placeholders

import java.util.{Collection => JavaCollection, Map => JavaMap}

import com.xebialabs.deployit.plugin.api.reflect.{Descriptor, DescriptorRegistry, PropertyKind}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.sql.base.idToPath

import scala.jdk.CollectionConverters._
import scala.language.implicitConversions

class PlaceholderHelper() {
  private val patternPlaceholders = """(?<=\{\{).*?(?=\}\})""".r

  private def getPlaceholderPatternMatch(propValue: AnyRef): Seq[String] = {
    patternPlaceholders.findAllIn(if (propValue == null) "" else propValue.toString).toList
  }

  def getPlaceholderEntriesFromCis(cis: List[ConfigurationItem]): List[PlaceholderEntry] = {
    cis.foldLeft(List[PlaceholderEntry]()) {
      (acc, ci) => {
        val descriptor: Descriptor = DescriptorRegistry.getDescriptor(ci.getType)
        val placeholderNames = getPlaceholderNamesFromCi(descriptor, ci)
        val path = idToPath(ci.getId)
        acc ++ placeholderNames.map(placeholderName => PlaceholderEntry(placeholderName, ci.getName, path, ci.getType)).toList
      }
    }
  }

  def getPlaceholderNamesFromCi(descriptor: Descriptor, ci: ConfigurationItem): Set[String] = {
    val descriptors = descriptor.getPropertyDescriptors.asScala
    val namePlaceholders = getPlaceholderPatternMatch(ci.getName).toSet

    (descriptors.flatMap(propertyDescriptor => {
      if (propertyDescriptor.getName.equals("placeholders")) {
        if (ci.getProperty(propertyDescriptor.getName).isInstanceOf[JavaCollection[_ <: String]]) {
          ci.getProperty(propertyDescriptor.getName).asInstanceOf[JavaCollection[String]].asScala.toSet
        } else {
          Set.empty
        }
      } else {
        propertyDescriptor.getKind match {
          case PropertyKind.BOOLEAN |
               PropertyKind.ENUM |
               PropertyKind.DATE |
               PropertyKind.INTEGER |
               PropertyKind.STRING =>
            val propValue = ci.getProperty(propertyDescriptor.getName).asInstanceOf[AnyRef]
            getPlaceholderPatternMatch(propValue)
          case PropertyKind.CI =>
            val propValue = ci.getProperty(propertyDescriptor.getName).asInstanceOf[ConfigurationItem]
            if (propValue == null) List.empty else getPlaceholderPatternMatch(propValue.getId)
          case PropertyKind.LIST_OF_STRING | PropertyKind.SET_OF_STRING =>
            val propValues = ci.getProperty(propertyDescriptor.getName).asInstanceOf[JavaCollection[String]]
            if (propValues == null) List.empty else propValues.asScala.flatMap(propValue => getPlaceholderPatternMatch(propValue, propertyDescriptor.getName))
          case PropertyKind.LIST_OF_CI | PropertyKind.SET_OF_CI =>
            val propValues = ci.getProperty(propertyDescriptor.getName).asInstanceOf[JavaCollection[ConfigurationItem]]
            if (propValues == null) List.empty else propValues.asScala.flatMap(propValue => getPlaceholderPatternMatch(propValue))
          case PropertyKind.MAP_STRING_STRING =>
            val propValues = ci.getProperty(propertyDescriptor.getName).asInstanceOf[JavaMap[String, String]]
            if (propValues == null) List.empty else propValues.asScala.flatMap { case (_, value) => getPlaceholderPatternMatch(value) }
        }
      }
    }).toSet ++ namePlaceholders).filterNot(_.trim.isEmpty)
  }

  def getUniquePlaceholdersToSave(cisToParse: List[ConfigurationItem],
                                  getPlaceholdersFunction: List[String] => List[PlaceholderEntry]): List[PlaceholderEntry] =
    if (cisToParse.nonEmpty) {
      val ciPlaceholders = getPlaceholderEntriesFromCis(cisToParse)
      val ciIds = cisToParse.map(x => idToPath(x.getId))
      val existingPlaceholders = getPlaceholdersFunction(ciIds)
      ciPlaceholders.diff(existingPlaceholders)
    } else {
      List.empty
    }
}
