package com.xebialabs.deployit.repository.sql

import com.xebialabs.deployit.core.sql.spring.Setter
import com.xebialabs.deployit.core.sql.{SqlCondition => cond, SqlFunction => func, _}
import com.xebialabs.deployit.engine.api.dto.{DeploymentInfo, Ordering, Paging}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.HostRepository
import com.xebialabs.deployit.repository.sql.base.{idToPath, pathToId}
import com.xebialabs.deployit.repository.sql.persisters.EnvironmentMembersSchema
import com.xebialabs.deployit.security.sql.{CiResolver, SqlPermissionFilter}
import com.xebialabs.deployit.security.{PermissionChecker, RoleService}
import com.xebialabs.deployit.sql.base.schema.{CIS, CI_PROPERTIES}
import com.xebialabs.deployit.sql.base.schema.CIS.{name, path}
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository
import org.springframework.transaction.annotation.Transactional

import java.sql.ResultSet
import scala.jdk.CollectionConverters._

@Repository
@Transactional("mainTransactionManager")
class SqlHostRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                        @Autowired override val checker: PermissionChecker,
                        @Autowired override val roleService: RoleService,
                        @Autowired val ciResolver: CiResolver)
                       (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends HostQueries with SqlPermissionFilter with HostRepository {

  val DEFAULT_ORDERING: Ordering = new Ordering("name:asc")
  val DEPLOYED_APP_TYPE: Type = Type.valueOf("udm.DeployedApplication")

  override def findByEnvironment(envId: String, pattern: String, paging: Paging, order: Ordering): List[DeploymentInfo] = {
    val selectBuilder = new SelectFragmentBuilder(FIND_BY_ENVIRONMENT)
      .where(cond.equals(path.tableAlias("env"), idToPath(envId)))
      .where(cond.like(func.lower(name.tableAlias("host")), s"%${if (pattern == null) "" else pattern.toLowerCase}%"))

    addSorting(selectBuilder, order)
    addPaging(selectBuilder, paging)
    addReadPermission(selectBuilder, CIS.secured_ci.tableAlias("host"))

    val queryParams = if (selectBuilder.parameters == null) List(idToPath(envId))
    else List(idToPath(envId)) ++ selectBuilder.parameters

    jdbcTemplate.query(selectBuilder.query, Setter(queryParams), mapRowToDeploymentInfo).asScala.toList
  }

  override def countByEnvironment(envId: String, pattern: String): Int = {
    val selectBuilder = new SelectFragmentBuilder(COUNT_BY_ENVIRONMENT)
      .where(cond.equals(path.tableAlias("env"), idToPath(envId)))
      .where(cond.like(func.lower(name.tableAlias("host")), s"%${if (pattern == null) "" else pattern.toLowerCase}%"))

    addReadPermission(selectBuilder, CIS.secured_ci.tableAlias("host"))

    jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), mapCount).asScala.headOption.getOrElse(0)
  }

  private val mapCount: RowMapper[Int] = (rs: ResultSet, _) =>
    rs.getInt(1)

  private val mapRowToDeploymentInfo: RowMapper[DeploymentInfo] = (rs: ResultSet, _) => {
    val deploymentTime =
      if (rs.getTimestamp(HostAliases.lastDeploymentAt) == null) null
      else toDateTime(rs.getTimestamp(HostAliases.lastDeploymentAt, UTC_CALENDAR))

    new DeploymentInfo(
      pathToId(rs.getString(HostAliases.Host.ciPath)),
      Type.valueOf(rs.getString(HostAliases.Host.ciType)),
      deploymentTime,
      rs.getString(HostAliases.lastDeploymentBy)
    )
  }

  private def addPaging(selectBuilder: SelectBuilder, paging: Paging): SelectBuilder =
    if (paging != null) selectBuilder.showPage(paging.page, paging.resultsPerPage) else selectBuilder

  private def addSorting(selectBuilder: SelectBuilder, ordering: Ordering): SelectBuilder = {
    val order = if (ordering == null) DEFAULT_ORDERING else ordering
    val columnToSort = SqlHostRepositoryFields.fieldMapping(order.field)
    selectBuilder.orderBy(
      if (order.isAscending) OrderBy.asc(columnToSort) else OrderBy.desc(columnToSort)
    )
  }

}

object HostAliases {

  object Host {
    val ciType = "host_ci_type"
    val ciPath = "host_ci_path"
    val ciName = "host_ci_name"
    val ciModifiedAt = "host_ci_modified_at"
  }

  object Environment {
    val ciPath = "env_ci_path"
  }

  val lastDeploymentAt = "lastDeploymentAt"
  val lastDeploymentBy = "lastDeploymentBy"

}

object SqlHostRepositoryFields {
  val NAME = "name"
  val MODIFIED_AT = "modifiedAt"
  val fieldMapping: Map[String, Selectable] = Map(
    NAME -> CIS.name.tableAlias("host"),
    MODIFIED_AT -> new SqlLiteral("lastDeployed.maxModifiedDate"),
  )
}

trait HostQueries extends Queries {
  val FIND_BY_ENVIRONMENT: String =
    sqlb"DISTINCT host.${CIS.path} ${HostAliases.Host.ciPath}, " +
      sqlb"host.${CIS.ci_type} ${HostAliases.Host.ciType}, " +
      sqlb"host.${CIS.name} ${HostAliases.Host.ciName}, " +
      sqlb"lastDeployed.maxModifiedDate ${HostAliases.lastDeploymentAt}," +
      sqlb"dd.${CIS.modified_by} ${HostAliases.lastDeploymentBy} " +
      sqlb"FROM ${CIS.tableName} env " +
      sqlb"INNER JOIN ${EnvironmentMembersSchema.tableName} em ON env.${CIS.ID} = em.${EnvironmentMembersSchema.environment_id} " +
      sqlb"INNER JOIN ${CIS.tableName} host ON em.${EnvironmentMembersSchema.member_id} = host.${CIS.ID} " +
      sqlb"LEFT JOIN (" +
      sqlb"SELECT d.${CIS.parent_id}, " +
      sqlb"max(d.${CIS.modified_at}) maxModifiedDate " +
      sqlb"FROM (" +
      sqlb"SELECT d.* FROM ${CIS.tableName} env " +
      sqlb"INNER JOIN ${CIS.tableName} da ON env.${CIS.ID} = da.${CIS.parent_id} " +
      sqlb"INNER JOIN ${CI_PROPERTIES.tableName} dap ON da.${CIS.ID} = dap.${CI_PROPERTIES.ci_id} " +
      sqlb"INNER JOIN ${CIS.tableName} d ON dap.${CI_PROPERTIES.ci_ref_value} = d.${CIS.ID} " +
      sqlb"WHERE env.${CIS.path} = ?" +
      sqlb") d " +
      sqlb"GROUP BY d.${CIS.parent_id}" +
      sqlb") lastDeployed ON host.${CIS.ID} = lastDeployed.${CIS.parent_id} " +
      sqlb"LEFT JOIN ${CIS.tableName} dd ON lastDeployed.maxModifiedDate = dd.${CIS.modified_at} AND host.${CIS.ID} = dd.${CIS.parent_id} "

  val COUNT_BY_ENVIRONMENT: String =
    sqlb"COUNT(host.${CIS.ID}) " +
      sqlb" FROM ${CIS.tableName} env " +
      sqlb"INNER JOIN ${EnvironmentMembersSchema.tableName} em ON env.${CIS.ID} = em.${EnvironmentMembersSchema.environment_id} " +
      sqlb"INNER JOIN ${CIS.tableName} host ON em.${EnvironmentMembersSchema.member_id} = host.${CIS.ID}"
}
