package com.xebialabs.deployit.repository.sql.placeholders

import java.sql.{PreparedStatement, ResultSet}

import com.xebialabs.deployit.core.sql._
import com.xebialabs.deployit.core.sql.spring.{Setter, XlSingleColumnRowMapper}
import com.xebialabs.deployit.engine.api.dto.{Ordering, Paging}
import com.xebialabs.deployit.plugin.api.deployment.{Placeholder, PlaceholderCi, PlaceholderCiWithValue}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.RepositoryService
import com.xebialabs.deployit.repository.placeholders.PlaceholderRepository
import com.xebialabs.deployit.repository.sql.base._
import com.xebialabs.deployit.repository.sql.placeholders.PlaceholdersSearchQueryBuilder.forPlaceholderPage
import com.xebialabs.deployit.security.RolesPermissionsPair
import com.xebialabs.deployit.security.sql.CiResolver
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{BatchPreparedStatementSetter, JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository
import org.springframework.transaction.annotation.Transactional
import com.xebialabs.deployit.core.sql.util._

import grizzled.slf4j.Logging

import java.sql.{PreparedStatement, ResultSet}
import java.util.UUID
import scala.jdk.CollectionConverters._
import scala.language.implicitConversions

@Repository
@Transactional("mainTransactionManager")
class SqlPlaceholderRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                               @Autowired val ciResolver: CiResolver,
                               @Autowired val repositoryService: RepositoryService)
                              (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo) extends PlaceholderRepository with PlaceholderQueries with Logging {

  private val placeholderHelper = new PlaceholderHelper()

  override def savePlaceholders(cis: List[ConfigurationItem]): Unit = {
    insertPlaceholders(placeholderHelper.getUniquePlaceholdersToSave(cis, getPlaceholders))
  }

  override def copyPlaceholders(fromPath: String, toPath: String): Unit = {
    jdbcTemplate.update(COPY, toPath, fromPath)
    jdbcTemplate.update(COPY_CHILDREN, toPath + "/", (fromPath.length + 2).asInstanceOf[Number], fromPath + "/%")
  }

  override def updatePlaceholders(cis: List[ConfigurationItem]): Unit = {
    val args = cis.map { ci =>
      val path = idToPath(ci.getId)
      Array(path.asInstanceOf[AnyRef])
    }
    jdbcTemplate.batchUpdate(DELETE, args.asJava)

    insertPlaceholders(placeholderHelper.getUniquePlaceholdersToSave(cis, getPlaceholders))
  }

  override def deletePlaceholders(ci: ConfigurationItem): Unit = {
    val start = System.currentTimeMillis()
    val path = idToPath(ci.getId)
    val deletedCount = jdbcTemplate.update(DELETE_WITH_CHILDREN, path, path + "/%")
    val end = System.currentTimeMillis()
    logger.debug(s"time taken to delete ${deletedCount} placeholder rows is : ${end - start}")
  }

  override def deletePlaceholders(cis: List[ConfigurationItem]): Unit = {
    val start = System.currentTimeMillis()
    val args = cis.map { ci =>
      val path = idToPath(ci.getId)
      Array(path.asInstanceOf[AnyRef])
    }
    val deletedCount = jdbcTemplate.batchUpdate(DELETE, args.asJava).toList
    val end = System.currentTimeMillis()
    logger.debug(s"time taken to delete ${deletedCount.sum} placeholder rows is : ${end - start}")
  }

  override def updatePath(oldPath: String, newPath: String): Unit = {
    jdbcTemplate.update(UPDATE_PATH, newPath, oldPath)
    jdbcTemplate.update(UPDATE_CHILDREN_PATH, newPath + "/", (oldPath.length + 2).asInstanceOf[Number], oldPath + "/%")
  }

  override def getPlaceholders(key: String, applicationId: String,
                               applicationName: String, rolesPermissions: Option[RolesPermissionsPair],
                               paging: Paging, order: Ordering): List[Placeholder] = {
    val selectBuilder = new PlaceholdersSearchQueryBuilder(Option(key), Option(applicationId), Option(applicationName),
      rolesPermissions, Option(order), paging).selectBuilder

    jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), new RowMapper[Placeholder] {
      override def mapRow(rs: ResultSet, rowNum: Int): Placeholder =
        Placeholder(
          rs.getInt(1),
          rs.getString(2),
          rs.getString(3),
          Type.valueOf(rs.getString(4)))
    }).asScala.toList
  }

  override def getPlaceholderKeys(key: String, rolesPermissions: Option[RolesPermissionsPair], paging: Paging, order: Ordering): List[String] = {
    val searchBuilder = new DistinctPlaceholdersSearchQueryBuilder(Option(key), rolesPermissions, Option(order), paging)
    if (rolesPermissions.isEmpty) {
      val selectBuilder = searchBuilder.selectBuilder
      Option(paging).map(p => forPlaceholderPage(selectBuilder, p.getPage, p.getResultsPerPage))
      jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), new XlSingleColumnRowMapper(classOf[String])).asScala.toList
    } else {
      Option(paging).map(p => forPlaceholderPage(searchBuilder.placeholderBuilerWithPermission, p.getPage, p.getResultsPerPage))
      var keys = jdbcTemplate.query(searchBuilder.placeholderBuilerWithPermission.query,
        Setter(searchBuilder.placeholderBuilerWithPermission.parameters), new XlSingleColumnRowMapper(classOf[String])).asScala.toList

      Option(paging).map(p => forPlaceholderPage(searchBuilder.dictEntriesBuilderWithPermission, p.getPage, p.getResultsPerPage))
      keys = keys ::: jdbcTemplate.query(searchBuilder.dictEntriesBuilderWithPermission.query,
        Setter(searchBuilder.dictEntriesBuilderWithPermission.parameters), new XlSingleColumnRowMapper(classOf[String])).asScala.toList

      Option(paging).map(p => forPlaceholderPage(searchBuilder.dictEntriesBuilderWithPermission, p.getPage, p.getResultsPerPage))
      keys = keys ::: jdbcTemplate.query(searchBuilder.dictEncEntriesBuilderWithPermission.query,
        Setter(searchBuilder.dictEncEntriesBuilderWithPermission.parameters), new XlSingleColumnRowMapper(classOf[String])).asScala.toList

      keys.distinct.sorted
    }
  }

  override def getPlaceholdersWithDictionary(
                                              key: String, value: String, dictionaryId: String, dictionaryName: String,
                                              rolesPermissions: Option[RolesPermissionsPair],
                                              paging: Paging, order: Ordering): List[PlaceholderCiWithValue] = {
    val selectBuilder = new PlaceholdersDictionarySearchQueryBuilder(
      Option(key), Option(value), Option(dictionaryId), Option(dictionaryName), rolesPermissions, Option(order), paging).selectBuilder

    jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), new RowMapper[PlaceholderCiWithValue] {
      override def mapRow(rs: ResultSet, rowNum: Int): PlaceholderCiWithValue = {
        PlaceholderCiWithValue(
          PlaceholdersSearchQueryBuilder.unwrap(rs.getString(1)),
          rs.getString(2),
          rs.getString(3),
          Type.valueOf(rs.getString(4))
        )
      }
    }).asScala.toList
  }

  override def getPlaceholdersWithEnvironment(
                                               key: String, environmentId: String, environmentName: String,
                                               rolesPermissions: Option[RolesPermissionsPair],
                                               paging: Paging, order: Ordering): List[PlaceholderCi] = {
    val selectBuilder = new PlaceholdersEnvironmentSearchQueryBuilder(
      Option(key), None, Option(environmentId), Option(environmentName), rolesPermissions, Option(order), paging).selectBuilder

    jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), new RowMapper[PlaceholderCi] {
      override def mapRow(rs: ResultSet, rowNum: Int): PlaceholderCi =
        PlaceholderCi(rs.getString(1), Type.valueOf(rs.getString(2)))
    }).asScala.toList
  }

  override def countAllPlaceholders(key: String, applicationId: String, applicationName: String, rolesPermissions: Option[RolesPermissionsPair]): Int = {
    val builder = new PlaceholdersSearchCountQueryBuilder(Option(key), Option(applicationId), Option(applicationName), rolesPermissions).selectBuilder

    jdbcTemplate.query(builder.query, Setter(builder.parameters), (rs: ResultSet, _: Int) =>
      rs.getInt(1)).asScala.headOption.getOrElse(0)
  }

  override def countPlaceholdersWithDictionary(key: String, value: String, dictionaryId: String, dictionaryName: String,
                                               rolesPermissions: Option[RolesPermissionsPair]): Int = {
    val builder = new PlaceholdersDictionarySearchCountQueryBuilder(
      Option(key), Option(value), Option(dictionaryId), Option(dictionaryName), rolesPermissions).selectBuilder


    jdbcTemplate.query(builder.query, Setter(builder.parameters), (rs: ResultSet, _: Int) =>
      rs.getInt(1)).asScala.headOption.getOrElse(0)
  }

  override def countPlaceholdersWithEnvironment(key: String, environmentId: String, environmentName: String,
                                                rolesPermissions: Option[RolesPermissionsPair]): Int = {
    val builder = new PlaceholdersEnvironmentSearchCountQueryBuilder(
      Option(key), None, Option(environmentId), Option(environmentName), rolesPermissions).selectBuilder


    jdbcTemplate.query(builder.query, Setter(builder.parameters), (rs: ResultSet, _: Int) =>
      rs.getInt(1)).asScala.headOption.getOrElse(0)
  }

  private def getPlaceholders(ids: List[String]): List[PlaceholderEntry] = {
    queryWithInClause(ids) { group =>
      val selectBuilder = new PlaceholdersSearchIdsQueryBuilder(group).selectBuilder
      jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), new RowMapper[PlaceholderEntry] {
        override def mapRow(rs: ResultSet, rowNum: Int): PlaceholderEntry =
          PlaceholderEntry(
            rs.getString(1),
            rs.getString(2),
            rs.getString(3),
            Type.valueOf(rs.getString(4)))
      }).asScala
    }
  }

  private def insertPlaceholders(placeholders: List[PlaceholderEntry]): Unit = {
    if (placeholders.nonEmpty) {
      batchInsert(INSERT, placeholders)
    }
  }

  private def batchInsert(query: String, placeholders: List[PlaceholderEntry]): Unit = {
    jdbcTemplate.batchUpdate(query, new BatchPreparedStatementSetter {
      override def getBatchSize: Int = placeholders.length

      override def setValues(ps: PreparedStatement, i: Int): Unit = {
        Setter.setString(ps, 1, placeholders(i).key)
        Setter.setString(ps, 2, placeholders(i).ciName)
        Setter.setString(ps, 3, placeholders(i).ciPath)
        Setter.setString(ps, 4, placeholders(i).ciType.toString)
      }
    })
  }

}

object SqlPlaceholderSelectQuery {
  val orderFieldMapping: Map[String, ColumnName] = Map(
    "key" -> PlaceholderSchema.key,
    "name" -> PlaceholderSchema.ciName,
    "ci_path" -> PlaceholderSchema.ciPath
  )
}

object PlaceholderSchema {
  val tableName: TableName = TableName("XLD_CI_PLACEHOLDERS")

  var id: ColumnName = ColumnName("ID")
  val key: ColumnName = ColumnName("key")
  val ciName: ColumnName = ColumnName("ci_name")
  val ciPath: ColumnName = ColumnName("ci_path")
  val ciType: ColumnName = ColumnName("ci_type")
}

trait PlaceholderQueries extends Queries {
  import PlaceholderSchema._

  lazy val INSERT: String = sqlb"insert into $tableName ($key, $ciName, $ciPath, $ciType) values (?,?,?,?)"
  lazy val COPY: String = sqlb"insert into $tableName ($key, $ciPath, $ciType) select $key, ?, $ciType from $tableName where $ciPath = ?"
  lazy val COPY_CHILDREN: String = sqlb"insert into $tableName ($key, $ciPath, $ciType) select $key, ${schemaInfo.sqlDialect.concat(schemaInfo.sqlDialect.paramWithCollation(), schemaInfo.sqlDialect.substr(ciPath))}, $ciType from $tableName where $ciPath like ?"
  lazy val DELETE: String = sqlb"delete from $tableName where $ciPath = ?"
  lazy val DELETE_WITH_CHILDREN: String = sqlb"delete from $tableName where $ciPath = ? or $ciPath like ?"
  lazy val UPDATE_PATH: String = sqlb"update $tableName set $ciPath = ? where $ciPath = ?"
  lazy val UPDATE_CHILDREN_PATH: String = sqlb"update $tableName set $ciPath = ${schemaInfo.sqlDialect.concat(schemaInfo.sqlDialect.paramWithCollation(), schemaInfo.sqlDialect.substr(ciPath))} where $ciPath like ?"
  lazy val DELETE_KEY: String = sqlb"delete from $tableName where $key = ? and $ciPath = ?"
}

case class PlaceholderEntry(key: String, ciName: String, ciPath: String, ciType: Type)
