package com.xebialabs.deployit.repository.sql

import com.xebialabs.deployit.core.api.dto.{ActiveUserSession, UserSessionsOverview}
import com.xebialabs.deployit.core.sql.spring.Setter
import com.xebialabs.deployit.core.sql.{JoinBuilder, _}
import com.xebialabs.deployit.engine.api.dto.{Ordering, Paging}
import com.xebialabs.deployit.engine.tasker.repository.sql.{PendingTaskMetadataSchema, PendingTaskSchema}
import com.xebialabs.deployit.sql.base.schema.{ActiveTaskMetadataSchema, ActiveTaskSchema}
import com.xebialabs.deployit.repository.XldSessionRepository
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.jdbc.core.{JdbcTemplate, ResultSetExtractor, SingleColumnRowMapper}
import org.springframework.stereotype.Repository
import org.springframework.transaction.annotation.Transactional

@Repository
@Transactional("mainTransactionManager")
class SqlXldSessionRepository(@Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                              @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends XldSessionRepository {

  import SpringSessionSchema._
  import com.xebialabs.deployit.security.sql.UserSchema

  val SESSION_DEFAULT_ORDERING: Ordering = new Ordering("principalName:asc")

  val deploymentTaskAlias = "deploymentTasks"
  val controlTaskAlias = "controlTasks"
  val usersAlias = "users"
  val usernameAlias = "username"
  val activeTasksAlias = "activeTasks"
  val activeTasksMetadataAlias = "activeTasksMeta"
  val pendingTasksAlias = "pendingTasks"
  val pendingTasksMetadataAlias = "pendingTasksMeta"
  val deploymentTaskTypes: String = DeploymentTaskType.values.toList.map("'" + _ + "'").toString().substring(4)
  val metadataValue: String = schemaInfo.sqlDialect.castToString(ActiveTaskMetadataSchema.metadata_value).build()
  val metadataKey: String = schemaInfo.sqlDialect.castToString(ActiveTaskMetadataSchema.metadata_key).build()

  val sessionFieldMapping: Map[String, ColumnName] = Map(
    "controlTasks" -> ColumnName(controlTaskAlias, quoted = false),
    "deploymentTasks" -> ColumnName(deploymentTaskAlias, quoted = false),
    "internal" -> ColumnName(usernameAlias, quoted = false),
    "principalName" -> principalName
  )

  override def findAll(principalPattern: String,
                       paging: Paging,
                       order: Ordering
                      ): List[ActiveUserSession] = {

    val unionBuilder = unionTasks(principalPattern)
    val findAllBuilder = addPaging(addSorting(selectAll(unionBuilder), sessionFieldMapping, order), paging)

    jdbcTemplate.query(
      findAllBuilder.query,
      Setter(unionBuilder.parameters),
      mapToSessions
    )
  }

  private def unionTasks(principalPattern: String): UnionBuilder = {
    new UnionBuilder(
      joinUsers(joinActiveTasks(selectUserTasks(principalPattern))),
      joinUsers(joinPendingTasks(selectUserTasks(principalPattern)))
    )
  }

  private def selectUserTasks(principalPattern: String): SelectBuilder = {
    withPrincipalFilter(principalPattern, selectUsername(selectTasksCount(selectUsers)))
      .groupBy(principalName)
      .groupBy(sessionId)
      .groupBy(UserSchema.USERNAME.tableAlias(usersAlias))
  }

  private def selectAll(sessionTaskJoinBuilder: AbstractQueryBuilder): SelectFragmentBuilder = {
    val query = s"t.${principalName.name} as ${principalName.name}, " +
      s"sum(t.$controlTaskAlias) as $controlTaskAlias, " +
      s"sum(t.$deploymentTaskAlias) as $deploymentTaskAlias, " +
      s"t.$usernameAlias as ${UserSchema.USERNAME.name} " +
      s"from (${sessionTaskJoinBuilder.query}) t " +
      s"group by t.${principalName.name}, t.$usernameAlias"
    new SelectFragmentBuilder(query)
  }

  override def count(principalPattern: String): Integer = {
    val selectBuilder = withPrincipalFilter(principalPattern, selectUsers.count())

    jdbcTemplate.query(
      selectBuilder.query,
      Setter(selectBuilder.parameters),
      new SingleColumnRowMapper[Number]
    ).get(0).intValue()
  }

  override def getSessionSummary: UserSessionsOverview = {
    val sessionOverviewSelectBuilder = selectOverview(
      new UnionBuilder(
        joinActiveTasks(
          selectTasksCount(selectUsers)
            .groupBy(principalName)
            .groupBy(sessionId)
        ),
        joinPendingTasks(
          selectTasksCount(selectUsers)
            .groupBy(principalName)
            .groupBy(sessionId)
        ))
    )

    jdbcTemplate.query(
      sessionOverviewSelectBuilder.query,
      Setter(sessionOverviewSelectBuilder.parameters),
      mapToSessionsOverview
    )
  }

  private def selectUsers = new SelectBuilder(springSessionTable)
    .select(principalName).distinct()

  private def selectTasksCount(selectBuilder: SelectBuilder) = {
    val controlTaskQuery = s"count(case when $metadataKey = 'taskType' AND " +
      s"$metadataValue NOT IN $deploymentTaskTypes THEN 1 else null end) as $controlTaskAlias"

    val deploymentTaskQuery =
      s"count(case when $metadataValue IN $deploymentTaskTypes THEN 1 else null end) as $deploymentTaskAlias"

    selectBuilder
      .select(new SqlLiteral(controlTaskQuery))
      .select(new SqlLiteral(deploymentTaskQuery))
  }

  private def selectUsername(selectBuilder: SelectBuilder): SelectBuilder = selectBuilder
    .select(UserSchema.USERNAME.tableAlias(usersAlias), usernameAlias)

  private def joinUsers(joinBuilder: JoinBuilder) = joinBuilder.join(
    new SelectBuilder(UserSchema.tableName).as(usersAlias),
    SqlCondition.equals(principalName, UserSchema.USERNAME.tableAlias(usersAlias)),
    JoinType.Left
  )

  private def joinActiveTasks(selectBuilder: SelectBuilder): JoinBuilder = new JoinBuilder(selectBuilder)
    .join(
      new SelectBuilder(ActiveTaskSchema.tableName).as(activeTasksAlias),
      SqlCondition.equals(principalName, ActiveTaskSchema.task_owner.tableAlias(activeTasksAlias)),
      JoinType.Left
    )
    .join(
      new SelectBuilder(ActiveTaskMetadataSchema.tableName).as(activeTasksMetadataAlias),
      SqlCondition.equals(ActiveTaskMetadataSchema.task_id.tableAlias(activeTasksMetadataAlias),
        ActiveTaskSchema.task_id.tableAlias(activeTasksAlias)),
      JoinType.Left
    )

  private def joinPendingTasks(selectBuilder: SelectBuilder): JoinBuilder = new JoinBuilder(selectBuilder)
    .join(
      new SelectBuilder(PendingTaskSchema.tableName).as(pendingTasksAlias),
      SqlCondition.equals(principalName, PendingTaskSchema.task_owner.tableAlias(pendingTasksAlias)),
      JoinType.Left
    ).join(
    new SelectBuilder(PendingTaskMetadataSchema.tableName).as(pendingTasksMetadataAlias),
    SqlCondition.equals(PendingTaskMetadataSchema.task_id.tableAlias(pendingTasksMetadataAlias),
      PendingTaskSchema.task_id.tableAlias(pendingTasksAlias)),
    JoinType.Left
  )

  private def selectOverview(sessionTaskJoinBuilder: UnionBuilder) = {
    val query = s"count(DISTINCT t.${principalName.name}) as totalActiveUsers," +
      s"sum(t.$controlTaskAlias) as totalControlTasks," +
      s"sum(t.$deploymentTaskAlias) as totalDeploymentTasks " +
      s"from (${sessionTaskJoinBuilder.query}) t"
    new SelectFragmentBuilder(query)
  }

  private def mapToSessions: ResultSetExtractor[List[ActiveUserSession]] = rs => {
    var sessions = List[ActiveUserSession]()

    while (rs.next) {
      val curr = new ActiveUserSession
      curr.setControlTasks(rs.getInt(controlTaskAlias))
      curr.setDeploymentTasks(rs.getInt(deploymentTaskAlias))
      curr.setUsername(rs.getString(principalName.name))
      curr.setInternal(rs.getString(UserSchema.USERNAME.name) != null)
      sessions ::= curr
    }

    sessions.reverse
  }

  private def mapToSessionsOverview: ResultSetExtractor[UserSessionsOverview] = rs => {
    val overview = new UserSessionsOverview
    while (rs.next) {
      overview.setTotalActiveUsers(rs.getInt("totalActiveUsers"))
      overview.setTotalControlTasks(rs.getInt("totalControlTasks"))
      overview.setTotalDeploymentTasks(rs.getInt("totalDeploymentTasks"))
    }
    overview
  }

  private def addPaging(builder: SelectBuilder, paging: Paging): SelectBuilder =
    if (paging != null) builder.showPage(paging.page, paging.resultsPerPage) else builder

  private def addSorting(selectBuilder: SelectBuilder, fieldMapping: Map[String, Selectable], ordering: Ordering, defaultOrdering: Ordering = SESSION_DEFAULT_ORDERING) = {
    val order = if (ordering == null) defaultOrdering else ordering
    val columnToSort = fieldMapping(order.field)
    if (order.field == SESSION_DEFAULT_ORDERING.field) {
      selectBuilder.orderBy(
        if (order.isAscending) OrderBy.asc(columnToSort) else OrderBy.desc(columnToSort)
      )
    } else {
      selectBuilder.orderBy(
        if (order.isAscending) OrderBy.asc(columnToSort) else OrderBy.desc(columnToSort)
      ).orderBy(OrderBy.asc(fieldMapping(SESSION_DEFAULT_ORDERING.field)))
    }

  }

  private def withPrincipalFilter(principalPattern: String, builder: SelectBuilder) = {
    if (principalPattern != null) {
      builder.where(SqlCondition.like(SqlFunction.lower(principalName), s"%$principalPattern%".toLowerCase()))
    }
    builder
  }
}

object SpringSessionSchema {
  val springSessionTable = TableName("SPRING_SESSION", quoted = false)

  val primaryId = ColumnName("PRIMARY_ID", quoted = false)
  val sessionId = ColumnName("SESSION_ID", quoted = false)
  val creationTime = ColumnName("CREATION_TIME", quoted = false)
  val lastAccessTime = ColumnName("LAST_ACCESS_TIME", quoted = false)
  val maxInactiveInterval = ColumnName("MAX_INACTIVE_INTERVAL", quoted = false)
  val principalName = ColumnName("PRINCIPAL_NAME", quoted = false)
}

object DeploymentTaskType extends Enumeration {
  type TaskType = Value
  val INITIAL, UPGRADE, UNDEPLOY, ROLLBACK, DEFAULT = Value
}
