package com.xebialabs.deployit.repository.sql

import com.xebialabs.deployit.core.api.dto.settings.ConfigurationDto
import com.xebialabs.deployit.core.sql.{ColumnName, Queries, SchemaInfo, TableName}
import com.xebialabs.deployit.repository.ConfigurationRepository
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.context.annotation.{Scope, ScopedProxyMode}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional

import java.sql.ResultSet
import scala.jdk.CollectionConverters.CollectionHasAsScala

@Component
@Scope(proxyMode = ScopedProxyMode.TARGET_CLASS)
@Transactional("mainTransactionManager")
class SqlConfigurationRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate)
                                (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends ConfigurationRepository with ConfigurationQueries {

  val keyPrefix: String = "deploy.configuration.setting.theme."

  override def saveOrUpdate(configurationSetting: ConfigurationDto): Unit = {
    if (isKeyExists(configurationSetting.key)) {
      jdbcTemplate.update(UPDATE, configurationSetting.value, keyPrefix + configurationSetting.key)
    } else {
      jdbcTemplate.update(INSERT, configurationSetting.title, keyPrefix + configurationSetting.key, configurationSetting.value)
    }
  }

  @Transactional(transactionManager = "mainTransactionManager", readOnly = true)
  override def getConfigurationByKey(): List[ConfigurationDto] = {
    jdbcTemplate.query(SELECT,
      new RowMapper[ConfigurationDto]() {
        override def mapRow(rs: ResultSet, rowNum: Int): ConfigurationDto = {
          ConfigurationDto(rs.getString(2), rs.getString(3), rs.getString(4))
        }
      }, "%setting.theme%").asScala.toList
  }

  override def isKeyExists(key: String): Boolean = {
    jdbcTemplate.queryForObject(CHECK_KEY_EXISTENCE, classOf[Integer], "%" + key) > 0
  }
}

object ConfigurationSchema {
  val tableName: TableName = TableName("XLD_CONFIGURATIONS")

  val id: ColumnName = ColumnName("ID")
  val title: ColumnName = ColumnName("title")
  val key: ColumnName = ColumnName("config_key")
  val value: ColumnName = ColumnName("config_value")
}


trait ConfigurationQueries extends Queries {

  import ConfigurationSchema._

  val SELECT = sqlb"select * from $tableName where $key like ?"

  val INSERT = sqlb"insert into $tableName ($title,$key, $value) values (?, ?, ?)"
  val UPDATE = sqlb"update $tableName set $value = ? where $key = ?"
  val CHECK_KEY_EXISTENCE = sqlb"select count(1) from $tableName where $key like ?"

}
