package com.xebialabs.deployit.repository.sql

import com.xebialabs.deployit.core.sql.spring.Setter
import com.xebialabs.deployit.core.sql.{OrderBy, Queries, SchemaInfo, SelectBuilder, SelectFragmentBuilder, Selectable, SimpleSqlFunction, SqlFunction, SqlLiteral, UTC_CALENDAR, toDateTime, SqlCondition => cond}
import com.xebialabs.deployit.engine.api.dto.{DeploymentInfo, Ordering, Paging}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.EnvironmentRepository
import com.xebialabs.deployit.repository.sql.CiConditions._
import com.xebialabs.deployit.repository.sql.base._
import com.xebialabs.deployit.repository.sql.persisters.EnvironmentMembersSchema
import com.xebialabs.deployit.security.sql.SqlPermissionFilter
import com.xebialabs.deployit.security.{PermissionChecker, RoleService}
import com.xebialabs.deployit.sql.base.schema._
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper, SingleColumnRowMapper}
import org.springframework.stereotype.Repository
import org.springframework.transaction.annotation.Transactional

import java.sql.ResultSet
import java.util
import scala.jdk.CollectionConverters._

@Repository
@Transactional("mainTransactionManager")
class SqlEnvironmentRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                               @Autowired override val roleService: RoleService,
                               @Autowired override val checker: PermissionChecker,
                               @Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends EnvironmentRepository
    with CiQueries
    with EnvironmentQueries
    with SqlPermissionFilter {

  val DEFAULT_PAGING: Paging = new Paging(1, 10)

  override def listEnvironmentsByHost(hostId: String,
                                      pattern: Option[String],
                                      paging: Paging,
                                      order: Ordering): util.List[DeploymentInfo] = {
    val selectBuilder = new SelectFragmentBuilder(SELECT_LATEST_DEPLOYMENT_FRAGMENT)
      .where(cond.equals(CIS.path.tableAlias("host"), idToPath(hostId)))

    addEnvironmentPattern(pattern, selectBuilder, "name")
    addReadPermission(selectBuilder, CIS.secured_ci.tableAlias("env"))

    val evaluatedPaging = if (paging == null) DEFAULT_PAGING else paging

    selectBuilder
      .orderBy(SqlEnvironmentsSelectQuery.toOrder(order)).showPage(evaluatedPaging.page, evaluatedPaging.resultsPerPage)

    val queryParameter =
      if (selectBuilder.parameters == null) List(idToPath(hostId))
      else List(idToPath(hostId)) ++ selectBuilder.parameters

    jdbcTemplate.query(
      selectBuilder.query,
      Setter(queryParameter),
      new RowMapper[DeploymentInfo] {
        override def mapRow(rs: ResultSet, rowNum: Int): DeploymentInfo = {
          val timestamp = rs.getTimestamp(3)
          val date = if (timestamp == null) null else toDateTime(rs.getTimestamp(3, UTC_CALENDAR))
          new DeploymentInfo(
            pathToId(rs.getString(1)),
            Type.valueOf(rs.getString(2)),
            date,
            rs.getString(4)
          )
        }
      }
    )
  }

  override def count(hostId: String, pattern: Option[String]): Int = {
    val selectEnvironment: SelectBuilder = createSelectEnvironmentIdsBuilder(hostId)

    val selectBuilder =
      new SelectBuilder(CIS.tableName).select(SqlFunction.countAll).where(cond.subselect(CIS.ID, selectEnvironment))

    addEnvironmentPattern(pattern, selectBuilder, null)
    addReadPermission(selectBuilder, CIS.secured_ci)

    jdbcTemplate.query(selectBuilder.query, Setter(selectBuilder.parameters), new SingleColumnRowMapper[Number]).get(0).intValue()
  }

  private def createSelectEnvironmentIdsBuilder(hostId: String) =
    new SelectBuilder(EnvironmentMembersSchema.tableName)
      .select(EnvironmentMembersSchema.environment_id).where(cond.subselect(EnvironmentMembersSchema.member_id, idToPK(hostId)))

  private def addEnvironmentPattern(environmentPattern: Option[String], selectBuilder: SelectBuilder, column: String) =
    environmentPattern match {
      case Some(pattern) if pattern.nonEmpty && column != null =>
        val dbColumn = SqlEnvironmentsSelectQuery.orderFieldMapping(column)(new Ordering("ASC"))
        selectBuilder.where(cond.like(dbColumn, s"%${pattern.toLowerCase}%"))
      case Some(pattern) if pattern.nonEmpty && column == null =>
        selectBuilder.where(cond.like(SqlFunction.lower(CIS.name), s"%${pattern.toLowerCase}%"))
      case _ =>
    }
}

object SqlEnvironmentsSelectQuery {
  val DEFAULT_SELECTABLE_FUNCTION: Ordering => SimpleSqlFunction = (_: Ordering) => SqlFunction.lower(CIS.name.tableAlias("env"))
  val orderFieldMapping: Map[String, Ordering => Selectable] = Map(
    "name" -> {
      DEFAULT_SELECTABLE_FUNCTION
    },
    "lastDeployment" -> { order =>
      new SqlLiteral(
        "case when latestDeployedApp.maxCreatedDate IS NULL THEN " +
          s"${if (order.isAscending) 1 else 0} ELSE ${if (order.isAscending) 0 else 1} END, latestDeployedApp.maxCreatedDate"
      )
    }
  )

  def toOrder(order: Ordering): OrderBy = {
    val DEFAULT_ORDERING: Ordering = new Ordering("name:asc")
    val evaluatedOrder: Ordering = if (order == null) DEFAULT_ORDERING else order
    val fields = SqlEnvironmentsSelectQuery.orderFieldMapping.keySet.asJava
    evaluatedOrder.validate(fields)
    val v: Option[Ordering => Selectable] = orderFieldMapping.get(evaluatedOrder.field)
    val selectable: Selectable = v.getOrElse(DEFAULT_SELECTABLE_FUNCTION)(evaluatedOrder)
    if (evaluatedOrder.isAscending) OrderBy.asc(selectable) else OrderBy.desc(selectable)
  }
}

trait EnvironmentQueries extends Queries {

  import com.xebialabs.deployit.sql.base.schema.{CI_PROPERTIES => cip}
  import com.xebialabs.deployit.repository.sql.persisters.{EnvironmentMembersSchema => em}
  import com.xebialabs.deployit.sql.base.schema.{CIS => ci}

  lazy val SELECT_LATEST_DEPLOYMENT_FRAGMENT: String = {
    sqlb"""env.${ci.path},
          |env.${ci.ci_type},
          |latestDeployedApp.maxCreatedDate,
          |d.${ci.modified_by},
          |env.${ci.name} as ci_name
          |FROM ${ci.tableName} env
          |INNER JOIN ${em.tableName} em ON env.${ci.ID} = em.${em.environment_id}
          |INNER JOIN ${ci.tableName} host ON em.${em.member_id} = host.${ci.ID}
          |LEFT JOIN (
          |SELECT da.${ci.parent_id}, MAX(da.${ci.modified_at}) as maxCreatedDate
          |FROM ${ci.tableName} da
          |INNER JOIN ${cip.tableName} dap ON da.${ci.ID} = dap.${cip.ci_id}
          |INNER JOIN ${ci.tableName} d ON dap.${cip.ci_ref_value} = d.${ci.ID}
          |INNER JOIN ${ci.tableName} host ON d.${ci.parent_id} = host.${ci.ID}
          |GROUP BY da.${ci.parent_id}, host.${ci.path}
          |HAVING host.${ci.path} = ?
          |) latestDeployedApp
          |ON (env.${ci.ID} = latestDeployedApp.${ci.parent_id})
          |LEFT JOIN ${ci.tableName} d ON env.${ci.ID} = d.${ci.parent_id} AND d.${ci.modified_at} = latestDeployedApp.maxCreatedDate"""
  }
}
