package com.xebialabs.deployit.repository.sql

import java.sql.Connection

import com.xebialabs.deployit.core.sql._
import com.xebialabs.deployit.core.sql.lock.{LockNameRegistry, SqlLockRepository}
import com.xebialabs.deployit.core.sql.spring.Setter.setString
import com.xebialabs.deployit.repository.ProfileRepository
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Repository
import org.springframework.transaction.annotation.Transactional

@Repository
@Transactional("mainTransactionManager")
class SqlProfileRepository(@Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                           @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends ProfileRepository with ProfileQueries with SqlLockRepository {

  private def createUserProfile(profile: Profile): Unit = {
    jdbcTemplate.update({ con: Connection =>
      val ps = con.prepareStatement(INSERT)
      setString(ps, 1, profile.username)
      setTimestamp(ps, 2, DateTime.now())
      ps
    })
  }

  override def updateProfile(profile: Profile): Unit = {
    jdbcTemplate.update({ con: Connection =>
      val ps = con.prepareStatement(UPDATE)
      setTimestamp(ps, 1, DateTime.now())
      setString(ps, 2, profile.username)
      ps
    })
  }

  override def updateLastLoginTime(profile: Profile): Unit = {
    if (userProfileExists(profile.username)) {
      updateProfile(profile)
    } else {
      executeWithLock(LockNameRegistry.XLD_PROFILES_LOCK) {
        if (userProfileExists(profile.username)) {
          updateProfile(profile)
        } else {
          createUserProfile(profile)
        }
      }
    }
  }

  override def userProfileExists(username: String): Boolean = {
    jdbcTemplate.queryForObject(CHECK_PROFILE_EXISTENCE, classOf[Integer], username) > 0
  }
}

case class Profile(username: String, lastLoginTime: DateTime)

object ProfileSchema {
  val tableName = TableName("XLD_PROFILES")

  val USERNAME = ColumnName("USERNAME")
  val LAST_LOGIN_TIME = ColumnName("LAST_LOGIN_TIME")
}

trait ProfileQueries extends Queries {
  import ProfileSchema._

  val SELECT = sqlb"select * from $tableName where $USERNAME = ?"
  val INSERT = sqlb"insert into $tableName ($USERNAME, $LAST_LOGIN_TIME) values (?, ?)"
  val UPDATE = sqlb"update $tableName set $LAST_LOGIN_TIME = ? where $USERNAME = ?"
  val CHECK_PROFILE_EXISTENCE = sqlb"select count(1) from $tableName where $USERNAME = ?"
}
