package com.xebialabs.deployit.repository.sql

import com.xebialabs.deployit.core.api.dto.ConfigurationVersion
import com.xebialabs.deployit.core.sql._
import com.xebialabs.deployit.repository.ConfigurationVersionRepository
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional

import java.sql.ResultSet
import scala.jdk.CollectionConverters._

@Component
@Transactional("mainTransactionManager")
class SqlConfigurationVersionRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate)
                                       (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends ConfigurationVersionRepository with ConfigurationVersionQueries {

  override def save(hash: String): Unit = {
    jdbcTemplate.update(INSERT, hash, toTimestamp(DateTime.now()))
  }

  private def configurationVersionMapper: RowMapper[ConfigurationVersion] = (rs: ResultSet, rowNum: Int) =>
    new ConfigurationVersion(rs.getInt(ConfigurationVersionSchema.id.name), rs.getString(ConfigurationVersionSchema.hash.name), toDateTime(rs.getTimestamp(ConfigurationVersionSchema.bootDateTime.name)))

  override def getCurrentConfigurationVersion: Option[ConfigurationVersion] =
    jdbcTemplate.query(SELECT_LAST, configurationVersionMapper).asScala.headOption
}

object ConfigurationVersionSchema {
  val tableName: TableName = TableName("XLD_CONFIG_VERSION")

  val id: ColumnName = ColumnName("ID")
  val hash: ColumnName = ColumnName("hash")
  val bootDateTime: ColumnName = ColumnName("boot_date_time")
}

trait ConfigurationVersionQueries extends Queries {

  import ConfigurationVersionSchema._

  val SELECT = sqlb"select $id, $hash, $bootDateTime from $tableName"
  val SELECT_LAST = sqlb"select $id, $hash, $bootDateTime from $tableName where $bootDateTime = (select max($bootDateTime) from $tableName)"
  val INSERT = sqlb"insert into $tableName ($hash, $bootDateTime) values (?, ?)"

}
