package com.xebialabs.deployit.repository.sql

import com.xebialabs.deployit.core.api.dto.BannerInfo
import com.xebialabs.deployit.core.api.dto.BannerInfo.BannerKind
import com.xebialabs.deployit.core.sql.{ColumnName, Queries, SchemaInfo, TableName}
import com.xebialabs.deployit.repository.BannerRepository
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional

import java.sql.ResultSet
import scala.jdk.CollectionConverters._

@Component
@Transactional("mainTransactionManager")
class SqlBannerRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate)
                         (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends BannerRepository with BannerQueries {

  override def update(bannerInfo: BannerInfo): Unit =
    if (jdbcTemplate.queryForObject(EXISTS, classOf[Number], bannerInfo.getKind.name()).intValue() == 0) {
      jdbcTemplate.update(INSERT, bannerInfo.getKind.name(), bannerInfo.getContent)
    } else {
      jdbcTemplate.update(UPDATE, bannerInfo.getContent, bannerInfo.getKind.name())
    }

  private def bannerMapper: RowMapper[BannerInfo] = (rs: ResultSet, rowNum: Int) =>
    new BannerInfo(BannerKind.valueOf(rs.getString(BannerSchema.kind.name)), rs.getString(BannerSchema.content.name))

  override def get(kind: BannerKind): Option[BannerInfo] =
    jdbcTemplate.query(SELECT, bannerMapper, kind.name()).asScala.headOption
}

object BannerSchema {
  val tableName: TableName = TableName("XLD_BANNER")

  val kind: ColumnName = ColumnName("kind")
  val content: ColumnName = ColumnName("content")
}

trait BannerQueries extends Queries {

  import BannerSchema._

  val SELECT = sqlb"select $kind, $content from $tableName where $kind = ?"
  val INSERT = sqlb"insert into $tableName ($kind, $content) values (?, ?)"
  val UPDATE = sqlb"update $tableName set $content = ? where $kind = ?"
  val EXISTS = sqlb"select count(*) from $tableName where $kind = ?"

}
