package com.xebialabs.deployit.core.upgrade.service

import com.xebialabs.deployit.core.sql.spring.{MapRowMapper, XlSingleColumnRowMapper}
import com.xebialabs.deployit.core.sql.{Queries, SchemaInfo}
import com.xebialabs.deployit.core.util.IdExtensions._
import com.xebialabs.deployit.repository.RepositoryService
import com.xebialabs.deployit.repository.sql.base.{CiPKType, asCiPKType, pathToId}
import com.xebialabs.deployit.sql.base.schema.CIS
import com.xebialabs.deployit.sql.base.schema.CIS._
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.dao.EmptyResultDataAccessException
import org.springframework.jdbc.core.{BatchPreparedStatementSetter, JdbcTemplate}
import org.springframework.stereotype.Component
import org.springframework.transaction.PlatformTransactionManager
import org.springframework.transaction.annotation.Transactional

import java.sql.PreparedStatement
import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

trait CiUpdateUpgraderService extends Queries {
  val rootType = "internal.Root"

  lazy val SELECT_CIS_WITH_NULL_PARENT_ID: String =
    sqlb"SELECT $ID, $path from $tableName " +
      sqlb"where $parent_id is null and $ci_type != '$rootType' order by $path"

  lazy val SELECT_ID_BY_PATH: String =
    sqlb"select $ID from $tableName where $path = ?"

  lazy val UPDATE_CIS_WITH_NULL_PARENT_ID: String = sqlb"UPDATE ${tableName} set $parent_id = ? " +
    sqlb"where $ID = ?"

  def updateOrDeleteCisWithNullParentId(): Unit
}

@Component
@Transactional("mainTransactionManager")
class DefaultCiUpdateUpgraderService(@Autowired val repositoryService: RepositoryService,
                                     @Autowired @Qualifier("mainSchema") val schemaInfo: SchemaInfo,
                                     @Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                     @Autowired @Qualifier("mainTransactionManager") val transactionManager: PlatformTransactionManager
                                    ) extends CiUpdateUpgraderService with Logging {

  def updateOrDeleteCisWithNullParentId(): Unit = {
    val ciIdMap = mutable.HashMap.empty[String, CiPKType]
    val cisWithNullParentId = fetchCis(SELECT_CIS_WITH_NULL_PARENT_ID)
    info(s"Found ${cisWithNullParentId.size} CIs with empty parent ID - starting refresh of parent ID column.")
    val cisToUpdate = cisWithNullParentId.flatMap(ci => {
      val parentPath = ci.path.getParent
      val parentIdMaybe = findCiParentId(parentPath, ciIdMap)
      parentIdMaybe match {
        case Some(parentId) =>
          ciIdMap.put(ci.path.getParent, parentId)
          debug(s"Adding parent to ${ci.path} - parent ID: $parentId.")
          Option(CiUpdateConfigurationItemProperties(ci.id, parentId))
        case _ =>
          debug(s"Couldn't resolve parent ID for ${ci.path}.")
          None
      }
    })

    if(cisToUpdate.nonEmpty) batchUpdate(UPDATE_CIS_WITH_NULL_PARENT_ID, cisToUpdate)

    val remainingCisWithNullParentId = fetchCis(SELECT_CIS_WITH_NULL_PARENT_ID)
    if(remainingCisWithNullParentId.nonEmpty) {
      info(s"Parent ID update finished, still found ${remainingCisWithNullParentId.size} CIs with null parent ID" +
        s" - starting removal of ${remainingCisWithNullParentId.map(_.path).mkString(", ")}")
      repositoryService.delete(remainingCisWithNullParentId.map(ci => pathToId(ci.path)): _*)
    }
  }

  private def findCiParentId(parentPath: String, ciIdMap: mutable.HashMap[String, CiPKType]): Option[CiPKType] = {
    Option(ciIdMap.getOrElseUpdate(parentPath, fetchCisId(parentPath)))
  }

  private def fetchCisId(parentPath: String): CiPKType =
    Try(jdbcTemplate.queryForObject(SELECT_ID_BY_PATH, new XlSingleColumnRowMapper(classOf[CiPKType]), parentPath)) match {
      case Success(id) => id
      case Failure(_: EmptyResultDataAccessException) => null
      case Failure(e) => throw e
    }

  private def fetchCis(query: String): List[CiUpdateConfigurationItem] =
    jdbcTemplate.query(query, MapRowMapper).asScala
      .map(
        m =>
          CiUpdateConfigurationItem(
            asCiPKType(m.get(CIS.ID.name)),
            m.get(CIS.path.name).asInstanceOf[String]
          )
      ).toList

  private def batchUpdate(query: String, cis: List[CiUpdateConfigurationItemProperties]): Unit = {
    if(cis.nonEmpty){
      jdbcTemplate.batchUpdate(query, new BatchPreparedStatementSetter {
        override def getBatchSize: Int = cis.length

        override def setValues(ps: PreparedStatement, i: Int): Unit = {
          ps.setInt(1, cis(i).parentId)
          ps.setInt(2, cis(i).id)
        }
      })
    }
  }
}

final case class CiUpdateConfigurationItem(id: Integer, path: String)
final case class CiUpdateConfigurationItemProperties(id: Integer, parentId: Integer);

