package com.xebialabs.deployit.core.upgrade

import com.xebialabs.deployit.core.sql.SchemaInfo
import com.xebialabs.deployit.core.upgrade.service.BatchUpgraderService
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.SearchParameters
import com.xebialabs.deployit.repository.placeholders.PlaceholderRepository
import com.xebialabs.deployit.repository.sql.CiRepository
import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.transaction.PlatformTransactionManager

import scala.jdk.CollectionConverters._

class Deployit900CiPlaceholderMigrationUpgrader
(
  @Autowired @Qualifier("mainSchema") val mainSchemaInfo: SchemaInfo,
  @Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate,
  @Autowired @Qualifier("mainTransactionManager") transactionManager: PlatformTransactionManager,
  @Autowired ciRepository: CiRepository,
  @Autowired placeholderRepository: PlaceholderRepository,
  @Autowired batchUpgraderService: BatchUpgraderService
) extends Upgrade with Logging {

  override def doUpgrade(): Boolean = {
    val query = new SearchParameters().setAncestor("Applications")
    batchUpgraderService.getBatchCis(query, params => {
      val existingTypeCiIds = ciRepository.list(params).asScala
        .flatMap {
          ciData =>
            if (ciData.getType.exists()) {
              Some(ciData.getId)
            } else {
              logger.warn(s"Skipping placeholder migration for '${ciData.getId}'. Unknown type: '${ciData.getType}'.")
              None
            }
        }
      val cis = existingTypeCiIds.flatMap(readCiOrCatchError).toList
      placeholderRepository.savePlaceholders(cis)
    })
    true
  }

  private def readCiOrCatchError(id: String): Option[ConfigurationItem] = {
    try {
      Some(ciRepository.read(id, null, useCache = false, depth = 1))
    } catch {
      case e: Exception =>
        logger.error(s"Unable to read ConfigurationItem[$id]. Reason: ${e.getMessage}")
        None
    }
  }

  override def upgradeVersion(): Version = Version.valueOf("deployit", "9.0.0")
}
