package com.xebialabs.deployit.core.upgrade

import ai.digital.deploy.core.common.security.permission.{DeployitPermissions, StitchPermissions}
import com.xebialabs.deployit.core.sql.spring.transactional
import com.xebialabs.deployit.core.upgrade.Deployit1000ReadOnlyRoleUpgrader.READ_ONLY_ROLE_NAME
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.deployit.security.permission.PlatformPermissions.{LOGIN, REPORT_VIEW, VIEW_SECURITY}
import com.xebialabs.deployit.security.service.UserProfileService
import com.xebialabs.deployit.security.{PermissionEditor, Role, RoleService, UserService}
import com.xebialabs.deployit.server.api.upgrade.{RepositoryInitialization, Upgrade, Version}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.transaction.PlatformTransactionManager

import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

class Deployit1000ReadOnlyRoleUpgrader(
                                        @Autowired roleService: RoleService,
                                        @Autowired xldUserProfileService: UserProfileService,
                                        @Autowired permissionEditor: PermissionEditor,
                                        @Autowired userService: UserService,
                                        @Autowired @Qualifier("mainTransactionManager") transactionManager: PlatformTransactionManager
                                      ) extends Upgrade with RepositoryInitialization with Logging {

  override def doUpgrade(): Boolean = doCreateRoleAndProfiles()

  override def upgradeVersion(): Version = Version.valueOf("deployit", "10.0.0")

  override def doInitialize(): Unit = doCreateRoleAndProfiles()

  override def getComponent: String = upgradeVersion().getComponent

  override def getPriority: Integer = Integer.MAX_VALUE

  private def doCreateUserProfiles(): Boolean = {
    Try({
      transactional(transactionManager) {
        createProfiles()
        logger.debug("User profiles created.")
      }
    }) match {
      case Success(_) =>
        true
      case Failure(exception: Exception) =>
        logger.error(s"Unable to complete ${getClass.getName}.", exception)
        false
    }
  }

  private def createProfiles(): Unit = {
    userService.listUsernames().asScala foreach (username => {
      Try(xldUserProfileService.findOne(username)) match {
        case Success(_) => identity()
        case Failure(_: NotFoundException) =>
          xldUserProfileService.createOrUpdate(username)
        case Failure(e) => throw e
      }
    })
  }

  private def doCreateRole(): Boolean = {
    Try({
      if (!roleAlreadyExists()) {
        transactional(transactionManager) {
          createReadOnlyRole()
          logger.debug(s"""Read only role "$READ_ONLY_ROLE_NAME" created.""")
        }
      }
      assignPermissions()
    }) match {
      case Success(_) =>
        true
      case Failure(exception: Exception) =>
        logger.error("Unable to complete Deployit1000ReadOnlyRoleUpgrader.", exception)
        false
    }
  }

  private def createReadOnlyRole(): Unit = {
    roleService.create(READ_ONLY_ROLE_NAME)
  }

  private def assignPermissions(): Unit = {
    implicit val role: Role = roleService.getRoleForRoleName(READ_ONLY_ROLE_NAME)
    createAndSavePermissions("global", Set(
      VIEW_SECURITY,
      LOGIN,
      REPORT_VIEW,
      DeployitPermissions.TASK_VIEW,
      DeployitPermissions.TASK_PREVIEWSTEP,
      StitchPermissions.VIEW
    ))
  }

  private def roleAlreadyExists(): Boolean = {
    try {
      roleService.getRoleForRoleName(READ_ONLY_ROLE_NAME)
      logger.warn(s"A role with name $READ_ONLY_ROLE_NAME already exists. Skipping role creation in ${this.getClass.getSimpleName}. " +
        "Recommend manually creating an admin read only role with all functional read permissions.")
      true
    } catch {
      case _: NotFoundException => false
    }
  }

  private def createPermissions(ci: String, permissions: Set[Permission])(implicit role: Role): java.util.Map[Role, java.util.Set[Permission]] = {
    val permissionsMap = permissionEditor.readPermissions(ci)
    permissionsMap.put(role, permissions.asJava)
    permissionsMap
  }

  private def createAndSavePermissions(ci: String, permissions: Set[Permission])(implicit role: Role): Unit = {
    val permissionsMap = createPermissions(ci, permissions)
    permissionEditor.editPermissions(ci, permissionsMap)
  }

  private def doCreateRoleAndProfiles(): Boolean = doCreateUserProfiles && doCreateRole
}

object Deployit1000ReadOnlyRoleUpgrader {
  val READ_ONLY_ROLE_NAME = "deploy_admin_read_only"
}
